/*
 * sag_68692.c -- serial driver for for option card pr55xx
 *
 * Copyright (C) 2005 Andreas Horn <andreas.horn@sartorius.com>
 */

#include <linux/module.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/wait.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/string.h>
#include <linux/fcntl.h>
#include <linux/mm.h>
#include <linux/kernel.h>
#include <linux/serial.h>
#include <linux/serialP.h>
#include <linux/console.h>
#include <linux/init.h>
#include <linux/bitops.h>
#include <linux/delay.h>
#include <linux/pm.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/system.h>
#include <asm/segment.h>
#include <asm/semaphore.h>
#include <asm/delay.h>
#include <asm/coldfire.h>
#include <asm/mcfsim.h>
#include <asm/sag_duart.h>
#include <asm/uaccess.h>
#include <asm/mcfedgeport.h>
#include "sag_68692.h"

#ifdef MCFPP_DCD0
struct timer_list duart_timer_struct;
#endif
/*
 *	Driver data structures.
 */
static struct tty_driver *duart_serial_driver;

/* Debugging...
 */
// #define SERIAL_DEBUG
// #define SERIAL_DEBUG_OPEN



#define	IRQBASE	 MCFINT_VECBASE

/* make a defined delay of about 900 ns using a non used CS with 15 waitstates */
#define DELAY *(volatile char*)(0x80030000)=00;\
		*(volatile char*)(0x80030000)=00;\
		*(volatile char*)(0x80030000)=00;
/*
 *	Configuration table, UARTs to look for at startup.
 */
#if  defined(CONFIG_SAG_PR5800_CORE)
static struct sag_serial duart_table[] = {
	{  /* ttyS3 */
		.magic = 0,
		.addr = (volatile unsigned char *) (DUART1_BASE),
		.chanaddr = (volatile unsigned char *) (DUART1_BASE),
		.irq = IRQBASE+1,
		.flags = ASYNC_UART_OFFLINE,
	},
	{  /* ttyS4 */
		.magic = 0,
		.addr = (volatile unsigned char *) (DUART1_BASE),
		.chanaddr = (volatile unsigned char *) (DUART1_BASE+8),
		.irq = IRQBASE+1,
		.flags = ASYNC_UART_OFFLINE,
	},
	{  /* ttyS5 */
		.magic = 0,
		.addr = (volatile unsigned char *) (DUART2_BASE),
		.chanaddr = (volatile unsigned char *) (DUART2_BASE),
		.irq = IRQBASE+2,
		.flags = ASYNC_UART_OFFLINE,
	},
	{  /* ttyS6 */
		.magic = 0,
		.addr = (volatile unsigned char *) (DUART2_BASE),
		.chanaddr = (volatile unsigned char *) (DUART2_BASE+8),
		.irq = IRQBASE+2,
		.flags = ASYNC_UART_OFFLINE,
	},
	{  /* ttyS7 */
		.magic = 0,
		.addr = (volatile unsigned char *) (DUART3_BASE),
		.chanaddr = (volatile unsigned char *) (DUART3_BASE),
		.irq = IRQBASE+3,
		.flags = ASYNC_UART_OFFLINE,
	},
	{  /* ttyS8 */
		.magic = 0,
		.addr = (volatile unsigned char *) (DUART3_BASE),
		.chanaddr = (volatile unsigned char *) (DUART3_BASE+8),
		.irq = IRQBASE+3,
		.flags = ASYNC_UART_OFFLINE,
	},
};
#endif
#if  defined(CONFIG_SAG_PR5410_CORE)
static struct sag_serial duart_table[] = {
	{  /* ttyS3 */
		.magic = 0,
		.addr = (volatile unsigned char *) (DUART1_BASE),
		.chanaddr = (volatile unsigned char *) (DUART1_BASE),
		.irq = IRQBASE+2,
		.flags = ASYNC_UART_OFFLINE,
	},
	{  /* ttyS4 */
		.magic = 0,
		.addr = (volatile unsigned char *) (DUART1_BASE),
		.chanaddr = (volatile unsigned char *) (DUART1_BASE+8),
		.irq = IRQBASE+2,
		.flags = ASYNC_UART_OFFLINE,
	},
	{  /* ttyS5 */
		.magic = 0,
		.addr = (volatile unsigned char *) (DUART2_BASE),
		.chanaddr = (volatile unsigned char *) (DUART2_BASE),
		.irq = IRQBASE+3,
		.flags = ASYNC_UART_OFFLINE,
	},
	{  /* ttyS6 */
		.magic = 0,
		.addr = (volatile unsigned char *) (DUART2_BASE),
		.chanaddr = (volatile unsigned char *) (DUART2_BASE+8),
		.irq = IRQBASE+3,
		.flags = ASYNC_UART_OFFLINE,
	},
};
#endif

#define	NR_PORTS	(sizeof(duart_table) / sizeof(struct sag_serial))

#ifdef CONFIG_PM
static struct pm_dev *serial_pm[NR_PORTS];
#endif



// return 0..255 on success
// return -1 on buserror

static int peek_byte(long addr)
{
	int ret=0;
	asm( 	" move.w %%sr,%%d2 ;"
		" move.w #0x2700,%%sr ;"
		" move.l _ramvec,%%a2 ;"
		" move.l 8(%%a2),%%d1 ;"
	        " lea   1f(%%pc),%%a1;"
	        " move.l %%a1,8(%%a2);"
	     	" move.l %%a7,%%a1;"
		" move.l #0,%0 ;"
		" move.b (%1),%0 ;"
		" bra.b  2f;"
		"1: move.l #-1,%0 ;"
		"2: move.l %%d1,8(%%a2) ;"
		" move.l %%a1,%%a7 ;"
		" move.w %%d2,%%sr ;"
		:"=d"(ret)
		:"a"(addr)
		:"%d1","%d2","%a1","%a2");
	return ret;
}

/*
 * tmp_buf is used as a temporary buffer by serial_write.  We need to
 * lock it in case the copy_from_user blocks while swapping in a page,
 * and some other program tries to do a serial write at the same time.
 * Since the lock will only come under contention when the system is
 * swapping and available memory is low, it makes sense to share one
 * buffer across all the serial ports, since it significantly saves
 * memory if large numbers of serial ports are open.
 */
//static unsigned char duart_tmp_buf[4096]; /* This is cheating */
//static DECLARE_MUTEX(duart_tmp_buf_sem);

/*
 *	Forware declarations...
 */
static void	duart_change_speed(struct sag_serial *info);
static void	duart_wait_until_sent(struct tty_struct *tty, int timeout);


static inline int serial_paranoia_check(struct sag_serial *info,
					char *name, const char *routine)
{
#ifdef SERIAL_PARANOIA_CHECK
	static const char badmagic[] =
		"duart(warning): bad magic number for serial struct %s in %s\n";
	static const char badinfo[] =
		"duart(warning): null sag_serial for %s in %s\n";

	if (!info) {
		printk(badinfo, name, routine);
		return 1;
	}
	if (info->magic != SERIAL_MAGIC) {
		printk(badmagic, name, routine);
		return 1;
	}
#endif
	return 0;
}

/*
 *	Sets or clears DTR and RTS on the requested line.
 */
static void duart_setsignals(struct sag_serial *info, int dtr, int rts)
{
	volatile unsigned char	*uartp;
	unsigned long		flags;
	unsigned char signal;

	local_irq_save(flags);
	uartp = info->addr;
	if (dtr >= 0) {
		/* DTR is only possible on channel b */
		if (info->line&1){
			if(dtr) {
				uartp[DUART_UOP1] = DUART_UOP_DTRB;
				info->sigs |= TIOCM_DTR;
			}
			else {
				uartp[DUART_UOP0] = DUART_UOP_DTRB;
				info->sigs &= ~TIOCM_DTR;
			}
		}
	}
	if (rts >= 0) {
		if(info->line&1)
			signal = DUART_UOP_RTSB;
		else
			signal = DUART_UOP_RTSA;
		if (rts) {
			info->sigs |= TIOCM_RTS;
			uartp[DUART_UOP1] = signal;
		} else {
			info->sigs &= ~TIOCM_RTS;
			uartp[DUART_UOP0] = signal;
		}
	}
	local_irq_restore(flags);
	return;
}

/*
 *	Gets values of serial signals.
 */
static int duart_getsignals(struct sag_serial *info)
{
	volatile unsigned char	*uartp;
	volatile int		sigs=0;
	unsigned char signal;
	//local_irq_save(flags);
	uartp = info->addr;
	signal = uartp[DUART_UIPR];
	if(info->line&1)
	{
		sigs |= (signal & DUART_UIPR_CTSB) ? 0 : TIOCM_CTS;
		sigs |= (signal & DUART_UIPR_DCDB) ? 0 : TIOCM_CD;
		sigs |= (signal & DUART_UIPR_RIB) ? 0 : TIOCM_RNG;
	}
	else {
		sigs |= (signal & DUART_UIPR_CTSA) ? 0 : TIOCM_CTS;
	}
	sigs |= (info->sigs & (TIOCM_RTS|TIOCM_DTR));

	//local_irq_restore(flags);
	return(sigs);
}

/*
 * ------------------------------------------------------------
 * duart_stop() and duart_start()
 *
 * This routines are called before setting or resetting tty->stopped.
 * They enable or disable transmitter interrupts, as necessary.
 * ------------------------------------------------------------
 */
static void duart_stop(struct tty_struct *tty)
{
	volatile unsigned char	*uartp;
	struct sag_serial	*info = (struct sag_serial *)tty->driver_data;
	unsigned long		flags;

	if (serial_paranoia_check(info, tty->name, "duart_stop"))
		return;

	local_irq_save(flags);
	uartp = info->addr;
	if(info->line&1)
	{
		info--;
		info->imr &= ~DUART_UIR_TXREADYB;
	}
	else
		info->imr &= ~DUART_UIR_TXREADYA;
	uartp[DUART_UIMR] = info->imr;
	local_irq_restore(flags);
}

static void duart_start(struct tty_struct *tty)
{
	volatile unsigned char	*uartp;
	struct sag_serial	*info = (struct sag_serial *)tty->driver_data;
	unsigned long		flags;
	if (serial_paranoia_check(info, tty->name, "duart_start"))
		return;

	local_irq_save(flags);
	if (info->xmit_cnt && info->xmit_buf) {
		uartp = info->addr;

		info->chanaddr[DUART_UCR] = DUART_UCR_TXENABLE;
		if(info->flags & ASYNC_UART_AUTORTS)
		{
			duart_setsignals(info,-1,1);
		}
		if(info->line&1)
		{
			info--;
			info->imr |= DUART_UIR_TXREADYB;
		}
		else
			info->imr |= DUART_UIR_TXREADYA;
		uartp[DUART_UIMR] = info->imr;
	}
	local_irq_restore(flags);
}

/*
 * ----------------------------------------------------------------------
 *
 * Here starts the interrupt handling routines.  All of the following
 * subroutines are declared as inline and are folded into
 * duart_interrupt().  They were separated out for readability's sake.
 *
 * Note: duart_interrupt() is a "fast" interrupt, which means that it
 * runs with interrupts turned off.  People who may want to modify
 * duart_interrupt() should try to keep the interrupt handler as fast as
 * possible.
 * -----------------------------------------------------------------------
 */

static void receive_chars(struct sag_serial *info)
{
	volatile unsigned char	*uartp;
	struct tty_struct	*tty = info->tty;
	unsigned char		status, ch;
	char flg;

	if (!tty)
		return;

	uartp = info->chanaddr;

	while ((status = uartp[DUART_USR]) & DUART_USR_RXREADY) {

		/* the interrupt must be cleared everytime, even if the flipbuffer is full */
		ch = uartp[DUART_URB];
		info->stats.rx++;

		/* ignore char if flipbuffer is full */
		if (tty->flip.count >= TTY_FLIPBUF_SIZE)
			continue;

		flg=0;
		if (status & DUART_USR_RXERR)
			uartp[DUART_UCR] = DUART_UCR_CMDRESETERR;
		if (status & DUART_USR_RXBREAK) {
			info->stats.rxbreak++;
			flg= TTY_BREAK;
		} else if (status & DUART_USR_RXPARITY) {
			info->stats.rxparity++;
			flg = TTY_PARITY;
		} else if (status & DUART_USR_RXOVERRUN) {
			info->stats.rxoverrun++;
			flg= TTY_OVERRUN;
		} else if (status & DUART_USR_RXFRAMING) {
			info->stats.rxframing++;
			flg = TTY_FRAME;
		} else {
			flg = 0;
		}

		if(tty->ldisc.single_char_received) {
			tty->ldisc.single_char_received(tty,ch,flg);
			continue;
		}

		tty_insert_flip_char(tty, ch, flg);
	}
	if(tty->ldisc.single_char_received)
		return;
	schedule_work(&tty->flip.work);
	return;
}

static void transmit_chars(struct sag_serial *info)
{

	if (info->x_char) {
		/* Send special char - probably flow control */
		info->chanaddr[DUART_UTB] = info->x_char;
		info->x_char = 0;
		info->stats.tx++;
		return;
	}

	if ((info->xmit_cnt <= 0) || info->tty->stopped) {

		unsigned txempty=info->chanaddr[DUART_USR]&DUART_USR_TXEMPTY;
		/* automatic shut off RTS after finished send of the last character */
		info->chanaddr[DUART_UCR] = DUART_UCR_TXDISABLE;

		if(txempty && (info->flags & ASYNC_UART_AUTORTS))
		{
			duart_setsignals(info,-1,0);
		}

		DELAY
		if(info->line&1)
		{
			struct sag_serial *info1=info-1;
			info1->imr &= ~DUART_UIR_TXREADYB;
			info1->addr[DUART_UIMR] = info1->imr;
		}
		else
		{
			info->imr &= ~DUART_UIR_TXREADYA;
			info->addr[DUART_UIMR] = info->imr;
		}
		schedule_work(&info->tqueue);
		return;
	}

	//while (info->chanaddr[DUART_USR] & DUART_USR_TXREADY) {
		info->chanaddr[DUART_UTB] = info->xmit_buf[info->xmit_tail++];
		info->xmit_tail = info->xmit_tail & (SERIAL_XMIT_SIZE-1);
		info->stats.tx++;
		info->xmit_cnt--;
//		{
//			break;
//		}
//	}

	return;
}

/*
 * This is the serial driver's generic interrupt routine
 */
irqreturn_t duart_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct sag_serial	*info;
	unsigned char		isr;
	*(volatile unsigned char *)(MCF_MBAR + MCF_EPFR) = 1<<(irq-64); /* reset interrupt */
	info=(struct sag_serial*)dev_id;
#if 0
	switch(irq)
	{
#if  defined(CONFIG_SAG_PR5800_CORE)
		case IRQBASE+1: info = &duart_table[0]; break;
		case IRQBASE+2: info = &duart_table[2]; break;
		case IRQBASE+3: info = &duart_table[4]; break;
#endif
#if  defined(CONFIG_SAG_PR5410_CORE)
		case IRQBASE+2: info = &duart_table[0]; break;
		case IRQBASE+3: info = &duart_table[2]; break;
#endif
		default: 	return IRQ_NONE;
	}

	if((void*)info!=dev_id)
	{
		panic("duart %d irq info mismatch %08lX!=%08lX\n",irq,(long)(info),(long)(dev_id));
	}
#endif
	isr = info->addr[DUART_UISR];

	if (isr & DUART_UIR_RXREADYA){
		receive_chars(info+0);
	}
	if (isr & DUART_UIR_RXREADYB){
		receive_chars(info+1);
	}
	if (isr & DUART_UIR_TXREADYA){
		transmit_chars(info+0);
	}
	if (isr & DUART_UIR_TXREADYB){
		transmit_chars(info+1);
	}
	return IRQ_HANDLED;
}

/*
 * -------------------------------------------------------------------
 * Here ends the serial interrupt routines.
 * -------------------------------------------------------------------
 */

static void duart_offintr(void *private)
{
	struct sag_serial	*info = (struct sag_serial *) private;
	struct tty_struct	*tty;

	tty = info->tty;
	if (!tty)
		return;
	tty_wakeup(tty);
}

#ifdef MCFPP_DCD0

/*
 *	Change of state on a DCD line.
 */
void duart_modem_change(struct sag_serial *info, int dcd)
{
	if (info->count == 0)
		return;

	if (info->flags & ASYNC_CHECK_CD) {
		if (dcd)
			wake_up_interruptible(&info->open_wait);
		else
			schedule_work(&info->tqueue_hangup);
	}
}


unsigned short	duart_ppstatus;

/*
 * This subroutine is called when the RS_TIMER goes off. It is used
 * to monitor the state of the DCD lines - since they have no edge
 * sensors and interrupt generators.
 */
static void duart_timer(void)
{
	unsigned int	ppstatus, dcdval, i;

	ppstatus = mcf_getppdata() & (MCFPP_DCD0 | MCFPP_DCD1);

	if (ppstatus != duart_ppstatus) {
		for (i = 0; (i < 2); i++) {
			dcdval = (i ? MCFPP_DCD1 : MCFPP_DCD0);
			if ((ppstatus & dcdval) != (duart_ppstatus & dcdval)) {
				duart_modem_change(&duart_table[i],
					((ppstatus & dcdval) ? 0 : 1));
			}
		}
	}
	duart_ppstatus = ppstatus;

	/* Re-arm timer */
	duart_timer_struct.expires = jiffies + HZ/25;
	add_timer(&duart_timer_struct);
}

#endif	/* MCFPP_DCD0 */


/*
 * This routine is called from the scheduler tqueue when the interrupt
 * routine has signalled that a hangup has occurred. The path of
 * hangup processing is:
 *
 * 	serial interrupt routine -> (scheduler tqueue) ->
 * 	do_serial_hangup() -> tty->hangup() -> duart_hangup()
 *
 */
static void do_serial_hangup(void *private)
{
	struct sag_serial	*info = (struct sag_serial *) private;
	struct tty_struct	*tty;

	tty = info->tty;
	if (!tty)
		return;

	tty_hangup(tty);
}

static int startup(struct sag_serial * info)
{
	volatile unsigned char	*uartp;
	unsigned long		flags;

	if (info->flags & ASYNC_INITIALIZED)
		return 0;

	if (!info->xmit_buf) {
		info->xmit_buf = (unsigned char *) __get_free_page(GFP_KERNEL);
		if (!info->xmit_buf)
			return -ENOMEM;
	}

	local_irq_save(flags);

#ifdef SERIAL_DEBUG
	printk("starting up ttyS%d (irq %d)...\n", info->line, info->irq);
#endif

	/*
	 *	Reset UART, get it into known state...
	 */
	uartp = info->chanaddr;
	uartp[DUART_UCR] = DUART_UCR_CMDRESETRX;  /* reset RX */
	DELAY
	uartp[DUART_UCR] = DUART_UCR_CMDRESETTX;  /* reset TX */
	DELAY
	duart_setsignals(info, 0, 0);

	if (info->tty)
		clear_bit(TTY_IO_ERROR, &info->tty->flags);
	info->xmit_cnt = info->xmit_head = info->xmit_tail = 0;

	/*
	 * set the speed, bits and parity of the serial port
	 * enable the UART receiver
	 */
	duart_change_speed(info);

	/*
	 * interrupt only on receiver.
	 */
	if(info->line&1)
	{
		info--;
		info->imr |= DUART_UIR_RXREADYB;
		info->addr[DUART_UIMR] = info->imr;
		info++;
	}
	else{
		info->imr |= DUART_UIR_RXREADYA;
		info->addr[DUART_UIMR] = info->imr;
	}

	info->flags |= ASYNC_INITIALIZED;
	local_irq_restore(flags);
	return 0;
}

/*
 * This routine will shutdown a serial port; interrupts are disabled, and
 * DTR is dropped if the hangup on close termio flag is on.
 */
static void shutdown(struct sag_serial * info)
{
	volatile unsigned char	*uartp;
	unsigned long		flags;

	if (!(info->flags & ASYNC_INITIALIZED))
		return;

#ifdef SERIAL_DEBUG
	printk("Shutting down serial port %d (irq %d)....\n", info->line,
	       info->irq);
#endif

	local_irq_save(flags);

	uartp = info->chanaddr;
	if(info->line&1)
	{
		info--;
		info->imr &= ~DUART_UIR_TXREADYB;
		info->imr &= ~DUART_UIR_RXREADYB;
		info->addr[DUART_UIMR] = info->imr;  /* mask all interrupts */
		info++;
	}
	else
	{
		info->imr &= ~DUART_UIR_TXREADYA;
		info->imr &= ~DUART_UIR_RXREADYA;
		info->addr[DUART_UIMR] = info->imr;  /* mask all interrupts */
	}
	uartp[DUART_UCR] = DUART_UCR_CMDRESETRX;  /* reset RX */
	DELAY
	uartp[DUART_UCR] = DUART_UCR_CMDRESETTX;  /* reset TX */

	if (!info->tty || (info->tty->termios->c_cflag & HUPCL))
		duart_setsignals(info, 0, 0);

	if (info->xmit_buf) {
		free_page((unsigned long) info->xmit_buf);
		info->xmit_buf = 0;
	}

	if (info->tty)
		set_bit(TTY_IO_ERROR, &info->tty->flags);

	info->flags &= ~ASYNC_INITIALIZED;
	local_irq_restore(flags);
}


/*
 * This is used to figure out the divisor speeds and the timeouts.
 */
static int baud_table[] = {
	0, 50, 75, 110, 134, 150, 200, 300, 600, 1200, 1800, 2400, 4800,
	9600, 19200, 38400, 57600, 115200, 230400, 460800, 0
};

/*
 * This the table for the baudrate settings of the chip
 */
static int duart_baud_table[] = {
	0x00,		/* 75 not used */
	0x00,		/* 75 not used */
	0x00,		/* 75 not used */
	0x01,		/* 110 not used */
	0x02,		/* 134 not used */
	0x03,		/* 150 not used */
	0x03,		/* 200 not used*/
	0x44,		/* 300  */
	0x55,		/* 600  */
	0x66,		/* 1200 */
	0x66,		/* 1800 */
	0x88,		/* 2400 */
	0x99,		/* 4800 */
	0xBB,		/* 9600 */
	0xCC,		/* 19200*/
	0xCC,		/* 38400*/
	0
};
#define duart_BAUD_TABLE_SIZE \
			(sizeof(duart_baud_table)/sizeof(duart_baud_table[0]))

/*
 * This routine is called to set the UART divisor registers to match
 * the specified baud rate for a serial port.
 */
static void duart_change_speed(struct sag_serial *info)
{
	volatile unsigned char	*uartp;
	unsigned int		baudclk, cflag;
	unsigned long		flags;
	unsigned char		mr1, mr2;
	struct sag_serial 	*info2=info;
	int			i;

	if (!info->tty || !info->tty->termios)
		return;
	cflag = info->tty->termios->c_cflag;
	if (info->addr == 0)
		return;

	i = cflag & CBAUD;
	if (i & CBAUDEX) {
		i &= ~CBAUDEX;
		if (i < 1 || i > 4)
			info->tty->termios->c_cflag &= ~CBAUDEX;
		else
			i += 15;
	}
	if (i == 0) {
		duart_setsignals(info, 0, -1);
		return;
	}

	/* compute the baudrate clock */
	baudclk = duart_baud_table[i];

	info->baud = baud_table[i];

	mr1 = DUART_MR1_RXIRQRDY | DUART_MR1_RXERRCHAR;
	mr2 = 0;

	switch (cflag & CSIZE) {
	case CS5:	mr1 |= DUART_MR1_CS5; break;
	case CS6:	mr1 |= DUART_MR1_CS6; break;
	case CS7:	mr1 |= DUART_MR1_CS7; break;
	case CS8:
	default:	mr1 |= DUART_MR1_CS8; break;
	}

	if (cflag & PARENB) {
		if (cflag & CMSPAR) {
			if (cflag & PARODD)
				mr1 |= DUART_MR1_PARITYMARK;
			else
				mr1 |= DUART_MR1_PARITYSPACE;
		} else {
			if (cflag & PARODD)
				mr1 |= DUART_MR1_PARITYODD;
			else
				mr1 |= DUART_MR1_PARITYEVEN;
		}
	} else {
		mr1 |= DUART_MR1_PARITYNONE;
	}

	if (cflag & CSTOPB)
		mr2 |= DUART_MR2_STOP2;
	else
		mr2 |= DUART_MR2_STOP1;

	if (cflag & CRTSCTS) {
		mr1 |= DUART_MR1_RXRTS;
		mr2 |= DUART_MR2_TXCTS;
	}

	if (cflag & CLOCAL)
		info->flags &= ~ASYNC_CHECK_CD;
	else
		info->flags |= ASYNC_CHECK_CD;

	if(info->flags & ASYNC_UART_AUTORTS)
		mr2 |= DUART_MR2_TXRTS;

	uartp = info->chanaddr;

	local_irq_save(flags);
#ifdef SERIAL_DEBUG
	printk("%s(%d): mr1=%x mr2=%x baudclk=%x\n", __FILE__, __LINE__,
		mr1, mr2, baudclk);
#endif

	/* check if any of the 2 channels have 38400 baud */
	if(info->line&1)
		info2--;
	else
		info2++;
	if(info->baud == 38400 || info2->baud == 38400)
		info->addr[DUART_UACR] = 0x00;
	else
		info->addr[DUART_UACR] = 0x80;
	DELAY
	uartp[DUART_UCR] = DUART_UCR_CMDRESETRX;    	/* reset RX */
	DELAY
	uartp[DUART_UCR] = DUART_UCR_CMDRESETTX;    	/* reset TX */
	DELAY
	uartp[DUART_UCR] = DUART_UCR_CMDRESETMRPTR;	/* reset MR pointer */
	DELAY
	uartp[DUART_UMR] = mr1;
	DELAY
	uartp[DUART_UMR] = mr2;
	DELAY
	uartp[DUART_UCSR] = baudclk;			/* set baudrate */
	DELAY
	uartp[DUART_UCR] = DUART_UCR_RXENABLE;
	DELAY
	duart_setsignals(info, 0, (cflag & CRTSCTS)?1:0 );
	local_irq_restore(flags);
	return;
}

static void duart_flush_chars(struct tty_struct *tty)
{
	volatile unsigned char	*uartp;
	struct sag_serial	*info = (struct sag_serial *)tty->driver_data;
	unsigned long		flags;
	unsigned char mask;
	unsigned char imr;

	if (serial_paranoia_check(info, tty->name, "duart_flush_chars"))
		return;

	uartp = (volatile unsigned char *) info->addr;

	/*
	 * re-enable receiver interrupt
	 */
	local_irq_save(flags);
	if(info->line&1) mask = DUART_UIR_RXREADYB;
	else mask =DUART_UIR_RXREADYA;
	;

	imr = info->imr;
	if(info->line&1)
	{
		info--;
		imr = info->imr;
		info++;
	}


	if ((!(imr & mask)) &&
	    (info->flags & ASYNC_INITIALIZED) ) {
	    	if(info->line&1)
		{
			info--;
			info->imr |= mask;
			uartp[DUART_UIMR] = info->imr;
			info++;
		}
		else
		{
			info->imr |= mask;
			uartp[DUART_UIMR] = info->imr;
		}
	}
	local_irq_restore(flags);

	if (info->xmit_cnt <= 0 || tty->stopped || tty->hw_stopped ||
	    !info->xmit_buf)
		return;

	/* Enable transmitter */
	local_irq_save(flags);
	if(info->line&1)
	{
		--info;
		info->imr |= DUART_UIR_TXREADYB;
	}
	else info->imr |= DUART_UIR_TXREADYA;
	uartp[DUART_UIMR] = info->imr;
	local_irq_restore(flags);
}

static int duart_write(struct tty_struct * tty,
		    const unsigned char *buf, int count)
{
	volatile unsigned char	*uartp;
	struct sag_serial	*info = (struct sag_serial *)tty->driver_data;
	unsigned long		flags;
	int			c, total = 0;

#ifdef SERIAL_DEBUG
	printk("%s(%d): duart_write(tty=%x,buf=%x,count=%d)\n",
		__FILE__, __LINE__, (int)tty, (int)buf, count);
#endif

	if (serial_paranoia_check(info, tty->name, "duart_write"))
		return 0;

	/* if no option card */
	if(info->flags & ASYNC_UART_OFFLINE)
		return -ENODEV;

	if (!tty || !info->xmit_buf)
		return 0;

	local_save_flags(flags);
	while (1) {
		local_irq_disable();
		c = min(count, (int) min(((int)SERIAL_XMIT_SIZE) - info->xmit_cnt - 1,
			((int)SERIAL_XMIT_SIZE) - info->xmit_head));
		local_irq_restore(flags);

		if (c <= 0)
			break;

		memcpy(info->xmit_buf + info->xmit_head, buf, c);

		local_irq_disable();
		info->xmit_head = (info->xmit_head + c) & (SERIAL_XMIT_SIZE-1);
		info->xmit_cnt += c;
		local_irq_restore(flags);

		buf += c;
		count -= c;
		total += c;
	}

	local_irq_disable();
	info->chanaddr[DUART_UCR] = DUART_UCR_TXENABLE;
	if(info->flags & ASYNC_UART_AUTORTS )
		duart_setsignals(info,-1,1);
	uartp = info->addr;
	if(info->line&1)
	{
		--info;
		info->imr |= DUART_UIR_TXREADYB;
	}
	else info->imr |= DUART_UIR_TXREADYA;
	uartp[DUART_UIMR] = info->imr;

	local_irq_restore(flags);

	return total;
}

static int duart_write_room(struct tty_struct *tty)
{
	struct sag_serial *info = (struct sag_serial *)tty->driver_data;
	int	ret;

	if (serial_paranoia_check(info, tty->name, "duart_write_room"))
		return 0;
	ret = SERIAL_XMIT_SIZE - info->xmit_cnt - 1;
	if (ret < 0)
		ret = 0;
	return ret;
}

static int duart_chars_in_buffer(struct tty_struct *tty)
{
	struct sag_serial *info = (struct sag_serial *)tty->driver_data;

	if (serial_paranoia_check(info, tty->name, "duart_chars_in_buffer"))
		return 0;
	return info->xmit_cnt;
}

static void duart_flush_buffer(struct tty_struct *tty)
{
	struct sag_serial	*info = (struct sag_serial *)tty->driver_data;
	unsigned long		flags;

	if (serial_paranoia_check(info, tty->name, "duart_flush_buffer"))
		return;

	local_irq_save(flags);
	info->xmit_cnt = info->xmit_head = info->xmit_tail = 0;
	local_irq_restore(flags);

	tty_wakeup(tty);
}

/*
 * ------------------------------------------------------------
 * duart_throttle()
 *
 * This routine is called by the upper-layer tty layer to signal that
 * incoming characters should be throttled.
 * ------------------------------------------------------------
 */
static void duart_throttle(struct tty_struct * tty)
{
	struct sag_serial *info = (struct sag_serial *)tty->driver_data;
	volatile unsigned char	*uartp;

	if (serial_paranoia_check(info, tty->name, "duart_throttle"))
		return;

	if (I_IXOFF(tty)) {
		info->x_char = STOP_CHAR(tty);
		uartp = info->addr;
		if(info->line&1)
		{
			info--;
			info->imr &= ~DUART_UIR_TXREADYB;
		}
		else
			info->imr &= ~DUART_UIR_TXREADYA;
		uartp[DUART_UIMR] = info->imr;
	}

	/* Turn off RTS line (do this atomic) */
	if (tty->termios->c_cflag & CRTSCTS)
		duart_setsignals(info,-1,0);
}

static void duart_unthrottle(struct tty_struct * tty)
{
	struct sag_serial *info = (struct sag_serial *)tty->driver_data;
	volatile unsigned char	*uartp;
	unsigned long flags;
#ifdef SERIAL_DEBUG_THROTTLE
	char	buf[64];

	printk("unthrottle %s: %d....\n", _tty_name(tty, buf),
	       tty->ldisc.chars_in_buffer(tty));
#endif

	if (serial_paranoia_check(info, tty->name, "duart_unthrottle"))
		return;

	if (I_IXOFF(tty)) {
		if (info->x_char)
			info->x_char = 0;
		else {
			spin_lock_irqsave(&tty->read_lock, flags);
			info->x_char = START_CHAR(tty);
			uartp = info->addr;
			if (info->chanaddr[DUART_USR] & DUART_USR_TXREADY) {
				info->chanaddr[DUART_UTB] = info->x_char;
				info->x_char = 0;
			}
			if(info->line&1)
			{
				info--;
				info->imr &= ~DUART_UIR_TXREADYB;
			}
			else
				info->imr &= ~DUART_UIR_TXREADYA;
			uartp[DUART_UIMR] = info->imr;
			spin_unlock_irqrestore(&tty->read_lock, flags);
		}
	}

	/* Assert RTS line (do this atomic) */
	if (tty->termios->c_cflag & CRTSCTS)
		duart_setsignals(info,-1,1);
}

/*
 * ------------------------------------------------------------
 * duart_ioctl() and friends
 * ------------------------------------------------------------
 */

static int get_serial_info(struct sag_serial * info,
			   struct serial_struct * retinfo)
{
	struct serial_struct tmp;

	if (!retinfo)
		return -EFAULT;
	memset(&tmp, 0, sizeof(tmp));
	tmp.type = info->type;
	tmp.line = info->line;
	tmp.port = (unsigned int) info->addr;
	tmp.irq = info->irq;
	tmp.flags = info->flags;
	tmp.baud_base = info->baud_base;
	tmp.close_delay = info->close_delay;
	tmp.closing_wait = info->closing_wait;
	tmp.custom_divisor = info->custom_divisor;
	return copy_to_user(retinfo,&tmp,sizeof(*retinfo)) ? -EFAULT : 0;
}

static int set_serial_info(struct sag_serial * info,
			   struct serial_struct * new_info)
{
	struct serial_struct new_serial;
	struct sag_serial old_info;
	int 	retval = 0;

	if (!new_info)
		return -EFAULT;
	if (copy_from_user(&new_serial,new_info,sizeof(new_serial)))
		return -EFAULT;
	old_info = *info;

	if (!capable(CAP_SYS_ADMIN)) {
		if ((new_serial.baud_base != info->baud_base) ||
		    (new_serial.type != info->type) ||
		    (new_serial.close_delay != info->close_delay) ||
		    ((new_serial.flags & ~ASYNC_USR_MASK) !=
		     (info->flags & ~ASYNC_USR_MASK)))
			return -EPERM;
		info->flags = ((info->flags & ~ASYNC_USR_MASK) |
			       (new_serial.flags & ASYNC_USR_MASK));
		info->custom_divisor = new_serial.custom_divisor;
		goto check_and_exit;
	}

	if (info->count > 1)
		return -EBUSY;

	/*
	 * OK, past this point, all the error checking has been done.
	 * At this point, we start making changes.....
	 */

	info->baud_base = new_serial.baud_base;
	info->flags = ((info->flags & ~ASYNC_FLAGS) |
			(new_serial.flags & ASYNC_FLAGS));
	info->type = new_serial.type;
	info->close_delay = new_serial.close_delay;
	info->closing_wait = new_serial.closing_wait;

check_and_exit:
	retval = startup(info);
	return retval;
}

/*
 * get_lsr_info - get line status register info
 *
 * Purpose: Let user call ioctl() to get info when the UART physically
 * 	    is emptied.  On bus types like RS485, the transmitter must
 * 	    release the bus after transmitting. This must be done when
 * 	    the transmit shift register is empty, not be done when the
 * 	    transmit holding register is empty.  This functionality
 * 	    allows an RS485 driver to be written in user space.
 */
static int get_lsr_info(struct sag_serial * info, unsigned int *value)
{
	volatile unsigned char	*uartp;
	unsigned long		flags;
	unsigned char		status;

	local_irq_save(flags);
	uartp = info->chanaddr;
	status = (uartp[DUART_USR] & DUART_USR_TXEMPTY) ? TIOCSER_TEMT : 0;
	local_irq_restore(flags);

	return put_user(status,value);
}

/*
 * This routine sends a break character out the serial port.
 */
static void send_break(	struct sag_serial * info, int duration)
{
	volatile unsigned char	*uartp;
	unsigned long		flags;

	if (!info->addr)
		return;
	set_current_state(TASK_INTERRUPTIBLE);
	uartp = info->chanaddr;

	local_irq_save(flags);
	uartp[DUART_UCR] = DUART_UCR_TXENABLE;
	DELAY
	uartp[DUART_UCR] = DUART_UCR_CMDBREAKSTART;
	DELAY
	schedule_timeout(duration);
	uartp[DUART_UCR] = DUART_UCR_CMDBREAKSTOP;
	DELAY
	uartp[DUART_UCR] = DUART_UCR_TXDISABLE;
	local_irq_restore(flags);
}

static int duart_tiocmget(struct tty_struct *tty, struct file *file)
{
	struct sag_serial * info = (struct sag_serial *)tty->driver_data;

#ifdef SERIAL_DEBUG
	printk("duart_tiocmget\n");
#endif
	if (serial_paranoia_check(info, tty->name, "duart_ioctl"))
		return -ENODEV;
	if (tty->flags & (1 << TTY_IO_ERROR))
		return -EIO;

	return duart_getsignals(info);
}

static int duart_tiocmset(struct tty_struct *tty, struct file *file,
			  unsigned int set, unsigned int clear)
{
	struct sag_serial * info = (struct sag_serial *)tty->driver_data;
	int rts = -1, dtr = -1;

#ifdef SERIAL_DEBUG
	printk("duart_tiocmset set=%X clear=%X\n",set,clear);
#endif
	if (serial_paranoia_check(info, tty->name, "duart_ioctl"))
		return -ENODEV;
	if (tty->flags & (1 << TTY_IO_ERROR))
		return -EIO;

	if (set & TIOCM_RTS)
		rts = 1;
	if (set & TIOCM_DTR)
		dtr = 1;
	if (clear & TIOCM_RTS)
		rts = 0;
	if (clear & TIOCM_DTR)
		dtr = 0;

	duart_setsignals(info, dtr, rts);

	return 0;
}

static int duart_ioctl(struct tty_struct *tty, struct file * file,
		    unsigned int cmd, unsigned long arg)
{
	struct sag_serial * info = (struct sag_serial *)tty->driver_data;
	int retval, error;

	/* if no option card */
	if(info->flags & ASYNC_UART_OFFLINE)
		return -ENODEV;

	if (serial_paranoia_check(info, tty->name, "duart_ioctl"))
		return -ENODEV;

	if ((cmd != TIOCGSERIAL) && (cmd != TIOCSSERIAL) &&
	    (cmd != TIOCSERCONFIG) && (cmd != TIOCSERGWILD)  &&
	    (cmd != TIOCSERSWILD) && (cmd != TIOCSERGSTRUCT)) {
		if (tty->flags & (1 << TTY_IO_ERROR))
		    return -EIO;
	}

#ifdef SERIAL_DEBUG
	printk("duart_ioctl cmd=%X...\n", cmd);
#endif
	switch (cmd) {
		case TCSBRK:	/* SVID version: non-zero arg --> no break */
			retval = tty_check_change(tty);
			if (retval)
				return retval;
			tty_wait_until_sent(tty, 0);
			if (signal_pending(current))
				return -EINTR;
			if (!arg)
				send_break(info, HZ/4);	/* 1/4 second */
			return 0;
		case TCSBRKP:	/* support for POSIX tcsendbreak() */
			retval = tty_check_change(tty);
			if (retval)
				return retval;
			tty_wait_until_sent(tty, 0);
			send_break(info, arg ? arg*(HZ/10) : HZ/4);
			return 0;
		case TIOCGSOFTCAR:
			error = put_user(C_CLOCAL(tty) ? 1 : 0,
				    (unsigned long *) arg);
			if (error)
				return error;
			return 0;
		case TIOCSSOFTCAR:
			get_user(arg, (unsigned long *) arg);
			tty->termios->c_cflag =
				((tty->termios->c_cflag & ~CLOCAL) |
				 (arg ? CLOCAL : 0));
			return 0;
		case TIOCGSERIAL:
			error = verify_area(VERIFY_WRITE, (void *) arg,
						sizeof(struct serial_struct));
			if (error)
				return error;
			return get_serial_info(info,
					       (struct serial_struct *) arg);
		case TIOCSSERIAL:
			return set_serial_info(info,
					       (struct serial_struct *) arg);
		case TIOCSERGETLSR: /* Get line status register */
			error = verify_area(VERIFY_WRITE, (void *) arg,
				sizeof(unsigned int));
			if (error)
				return error;
			else
			    return get_lsr_info(info, (unsigned int *) arg);

		case TIOCSERGSTRUCT:
			error = copy_to_user((struct sag_serial *) arg,
				    info, sizeof(struct sag_serial));
			if (error)
				return -EFAULT;
			return 0;

		case TIOCSET422: {
			unsigned int val;
			get_user(val, (unsigned int *) arg);
			if(val)
				info->flags |= ASYNC_UART_AUTORTS;
			else
				info->flags &= ~ASYNC_UART_AUTORTS;
			duart_change_speed(info);
			break;
		}

		default:
			return -ENOIOCTLCMD;
		}
	return 0;
}

static void duart_set_termios(struct tty_struct *tty, struct termios *old_termios)
{
	struct sag_serial *info = (struct sag_serial *)tty->driver_data;

	if (tty->termios->c_cflag == old_termios->c_cflag)
		return;

#ifdef SERIAL_DEBUG
	printk("duart_set_termios\n");
#endif
	duart_change_speed(info);

	if ((old_termios->c_cflag & CRTSCTS) &&
	    !(tty->termios->c_cflag & CRTSCTS)) {
		tty->hw_stopped = 0;
		duart_setsignals(info, -1, 1);
#if 0
		duart_start(tty);
#endif
	}
}

/*
 * ------------------------------------------------------------
 * duart_close()
 *
 * This routine is called when the serial port gets closed.  First, we
 * wait for the last remaining data to be sent.  Then, we unlink its
 * S structure from the interrupt chain if necessary, and we free
 * that IRQ if nothing is left in the chain.
 * ------------------------------------------------------------
 */
static void duart_close(struct tty_struct *tty, struct file * filp)
{
	struct sag_serial	*info = (struct sag_serial *)tty->driver_data;
	unsigned long		flags;

	if (!info || serial_paranoia_check(info, tty->name, "duart_close"))
		return;

	local_irq_save(flags);

	if (tty_hung_up_p(filp)) {
		local_irq_restore(flags);
		return;
	}

#ifdef SERIAL_DEBUG_OPEN
	printk("duart_close ttyS%d, count = %d\n", info->line, info->count);
#endif
	if ((tty->count == 1) && (info->count != 1)) {
		/*
		 * Uh, oh.  tty->count is 1, which means that the tty
		 * structure will be freed.  Info->count should always
		 * be one in these conditions.  If it's greater than
		 * one, we've got real problems, since it means the
		 * serial port won't be shutdown.
		 */
		printk("duart: bad serial port count; tty->count is 1, "
		       "info->count is %d\n", info->count);
		info->count = 1;
	}
	if (--info->count < 0) {
		printk("duart: bad serial port count for ttyS%d: %d\n",
		       info->line, info->count);
		info->count = 0;
	}
	if (info->count) {
		local_irq_restore(flags);
		return;
	}
	info->flags |= ASYNC_CLOSING;

	/*
	 * Now we wait for the transmit buffer to clear; and we notify
	 * the line discipline to only process XON/XOFF characters.
	 */
	tty->closing = 1;
	tty_ldisc_flush(tty);	/* will call unthrottle if line is stopped now*/
	if (info->closing_wait != ASYNC_CLOSING_WAIT_NONE)
		tty_wait_until_sent(tty, info->closing_wait);

	/*
	 * At this point we stop accepting input.  To do this, we
	 * disable the receive line status interrupts, and tell the
	 * interrupt driver to stop checking the data ready bit in the
	 * line status register.
	 */
	if(info->line&1){
		--info;
		info->imr &= ~DUART_UIR_RXREADYB;
		info->addr[DUART_UIMR] = info->imr;
		info++;
	}
	else{
		info->imr &= ~DUART_UIR_RXREADYA;
		info->addr[DUART_UIMR] = info->imr;
	}

	shutdown(info);
	if (tty->driver->flush_buffer)
		tty->driver->flush_buffer(tty);

	tty->closing = 0;
	info->event = 0;
	info->tty = 0;

	if (info->blocked_open) {
		if (info->close_delay) {
			msleep_interruptible(jiffies_to_msecs(info->close_delay));
		}
		wake_up_interruptible(&info->open_wait);
	}
	info->flags &= ~(ASYNC_NORMAL_ACTIVE|ASYNC_CLOSING);
	wake_up_interruptible(&info->close_wait);
	local_irq_restore(flags);
}

/*
 * duart_wait_until_sent() --- wait until the transmitter is empty
 */
static void
duart_wait_until_sent(struct tty_struct *tty, int timeout)
{
	volatile unsigned char *uartp;
	unsigned long orig_jiffies, char_time=1;
	struct sag_serial * info = (struct sag_serial *)tty->driver_data;
	orig_jiffies = jiffies;
	char_time = (2 * HZ * 10 +info->baud-1) / info->baud;
	if (!timeout || timeout > 2)
		timeout = char_time;
#ifdef SERIAL_DEBUG
	printk("duart_wait_until_sent\n");
#endif
	uartp = (volatile unsigned char *) info->chanaddr;
	for(;;)
	{
		if (uartp[DUART_USR] & (DUART_USR_TXEMPTY))
			break;
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(char_time);
		if (signal_pending(current))
			break;
		if (timeout && time_after(jiffies, orig_jiffies + timeout))
			break;
	}
}

/*
 * duart_hangup() --- called by tty_hangup() when a hangup is signaled.
 */
void duart_hangup(struct tty_struct *tty)
{
	struct sag_serial * info = (struct sag_serial *)tty->driver_data;

	if (serial_paranoia_check(info, tty->name, "duart_hangup"))
		return;

	duart_flush_buffer(tty);
	shutdown(info);
	info->event = 0;
	info->count = 0;
	info->flags &= ~ASYNC_NORMAL_ACTIVE;
	info->tty = 0;
	wake_up_interruptible(&info->open_wait);
}

/*
 * ------------------------------------------------------------
 * duart_open() and friends
 * ------------------------------------------------------------
 */
static int block_til_ready(struct tty_struct *tty, struct file * filp,
			   struct sag_serial *info)
{
//	DECLARE_WAITQUEUE(wait, current);
//	int	retval;
//	int	do_clocal = 0;

	/*
	 * If the device is in the middle of being closed, then block
	 * until it's done, and then try again.
	 */
	if (info->flags & ASYNC_CLOSING) {
		interruptible_sleep_on(&info->close_wait);
#ifdef SERIAL_DO_RESTART
		if (info->flags & ASYNC_HUP_NOTIFY)
			return -EAGAIN;
		else
			return -ERESTARTSYS;
#else
		return -EAGAIN;
#endif
	}

	/* remove 2 wire rs422 mode */
	//info->flags &= ~ASYNC_UART_AUTORTS;

	/*
	 * If non-blocking mode is set, or the port is not enabled,
	 * then make the check up front and then exit.
	 */
	if ((filp->f_flags & O_NONBLOCK) ||
	    (tty->flags & (1 << TTY_IO_ERROR))) {
		info->flags |= ASYNC_NORMAL_ACTIVE;
		return 0;
	}

#if 0	// no wait for carried in SAG!!
	if (tty->termios->c_cflag & CLOCAL)
		do_clocal = 1;

	/*
	 * Block waiting for the carrier detect and the line to become
	 * free (i.e., not in use by the callout).  While we are in
	 * this loop, info->count is dropped by one, so that
	 * duart_close() knows when to free things.  We restore it upon
	 * exit, either normal or abnormal.
	 */
	retval = 0;
	add_wait_queue(&info->open_wait, &wait);
#ifdef SERIAL_DEBUG_OPEN
	printk("block_til_ready before block: ttyS%d, count = %d\n",
	       info->line, info->count);
#endif
	info->count--;
	info->blocked_open++;
	while (1) {
		local_irq_disable();
		duart_setsignals(info, 1, 1);
		local_irq_enable();
		current->state = TASK_INTERRUPTIBLE;
		if (tty_hung_up_p(filp) ||
		    !(info->flags & ASYNC_INITIALIZED)) {
#ifdef SERIAL_DO_RESTART
			if (info->flags & ASYNC_HUP_NOTIFY)
				retval = -EAGAIN;
			else
				retval = -ERESTARTSYS;
#else
			retval = -EAGAIN;
#endif
			break;
		}
		if (!(info->flags & ASYNC_CLOSING) &&
		    (do_clocal || (duart_getsignals(info) & TIOCM_CD)))
			break;
		if (signal_pending(current)) {
			retval = -ERESTARTSYS;
			break;
		}
#ifdef SERIAL_DEBUG_OPEN
		printk("block_til_ready blocking: ttyS%d, count = %d\n",
		       info->line, info->count);
#endif
		schedule();
	}
	current->state = TASK_RUNNING;
	remove_wait_queue(&info->open_wait, &wait);
	if (!tty_hung_up_p(filp))
		info->count++;
	info->blocked_open--;
#ifdef SERIAL_DEBUG_OPEN
	printk("block_til_ready after blocking: ttyS%d, count = %d\n",
	       info->line, info->count);
#endif
	if (retval)
		return retval;
#endif
	info->flags |= ASYNC_NORMAL_ACTIVE;
	return 0;
}

/*
 * This routine is called whenever a serial port is opened. It
 * enables interrupts for a serial port, linking in its structure into
 * the IRQ chain.   It also performs the serial-specific
 * initialization for the tty structure.
 */
int duart_open(struct tty_struct *tty, struct file * filp)
{
	struct sag_serial	*info;
	int 			retval, line;

	line = tty->index;
	if ((line < 0) || (line >= NR_PORTS))
		return -ENODEV;
	info = duart_table + line;
	/* if no option card */
	if(info->flags & ASYNC_UART_OFFLINE)
		return -ENODEV;

	if (serial_paranoia_check(info, tty->name, "duart_open"))
		return -ENODEV;
#ifdef SERIAL_DEBUG_OPEN
	printk("duart_open %s, count = %d\n", tty->name, info->count);
#endif
	info->count++;
	tty->driver_data = info;
	info->tty = tty;

	/*
	 * Start up serial port
	 */
	retval = startup(info);
	if (retval)
		return retval;

	retval = block_til_ready(tty, filp, info);
	if (retval) {
#ifdef SERIAL_DEBUG_OPEN
		printk("duart_open returning after block_til_ready with %d\n",
		       retval);
#endif
		return retval;
	}

#ifdef SERIAL_DEBUG_OPEN
	printk("duart_open %s successful...\n", tty->name);
#endif
	return 0;
}

/*
 *	Based on the line number set up the internal interrupt stuff.
 */
static void duart_irqinit(struct sag_serial *info)
{
	volatile unsigned char	*uartp;

	uartp = info->addr;

	/* interrupt request only once per duart */
	if(!(info->line&1))
	{
		if(enable_edgeport_dev(info->irq-IRQBASE,LEVEL_MODE,duart_interrupt,info)<0)
		printk( "%s(%d):Cant request interrupt, IRQ=%d\n",
			__FILE__,__LINE__,info->irq);
		return;
	}

	uartp = info->addr;

	/* Clear mask, so no surprise interrupts. */
	uartp[DUART_UIMR] = 0;

}


char *duart_drivername = "PR5510/04 UART serial driver version $Rev$\n";


/*
 * Serial stats reporting...
 */
int duart_readproc(char *page, char **start, off_t off, int count,
		         int *eof, void *data)
{
	struct sag_serial	*info;
	char			str[20];
	int			len, sigs, i;

	len = sprintf(page, duart_drivername);
	for (i = 0; (i < NR_PORTS); i++) {
		info = &duart_table[i];
		if(info->flags & ASYNC_UART_OFFLINE)
		{
			len += sprintf((page + len), "%d: port:%x irq=%d baud:%d Offline\n",
				i, (unsigned int) info->addr, info->irq, info->baud);
		}
		else
		{

			len += sprintf((page + len), "%d: port:%x irq=%d baud:%d ",
				i, (unsigned int) info->addr, info->irq, info->baud);
			if (info->stats.rx || info->stats.tx)
				len += sprintf((page + len), "tx:%d rx:%d ",
				info->stats.tx, info->stats.rx);
			if (info->stats.rxframing)
				len += sprintf((page + len), "fe:%d ",
				info->stats.rxframing);
			if (info->stats.rxparity)
				len += sprintf((page + len), "pe:%d ",
				info->stats.rxparity);
			if (info->stats.rxbreak)
				len += sprintf((page + len), "brk:%d ",
				info->stats.rxbreak);
			if (info->stats.rxoverrun)
				len += sprintf((page + len), "oe:%d ",
				info->stats.rxoverrun);

			str[0] = str[1] = 0;
			if ((sigs = duart_getsignals(info))) {
				if (sigs & TIOCM_RTS)
					strcat(str, "|RTS");
				if (sigs & TIOCM_CTS)
					strcat(str, "|CTS");
				if (sigs & TIOCM_DTR)
					strcat(str, "|DTR");
				if (sigs & TIOCM_CD)
					strcat(str, "|CD");
			}

			len += sprintf((page + len), "%s\n", &str[1]);
		}
	}

	return(len);
}

#ifdef CONFIG_PM
static struct pm_dev *serial_pm[NR_PORTS];
static int serial_pm_callback(struct pm_dev *dev, pm_request_t request, void *data)
{
	volatile unsigned long *imrp;
	//struct tty_struct *tty;
	struct sag_serial *info = (struct sag_serial *)dev->data;

	if(info == NULL)
		return -1;

	switch (request) {
	case PM_SUSPEND:
                /* disable interrupts */
		imrp=(volatile unsigned long *)(MCF_MBAR + MCFICM_INTC0 +MCFINTC_IMRL);
		*imrp |= (1 << (info->irq - 64));
		free_irq(info->irq,0);
		info->flags &= ~ASYNC_INITIALIZED;
		break;

	case PM_RESUME:
		/* restore interrupt settings */
		duart_irqinit(info);
                /* restore the uart settings: baudrate, bits, parity .. */
		startup(info);
		// signal a line discipline that a powerfail has occured
		if(info->tty && info->tty->ldisc.single_char_received) {
				info->tty->ldisc.single_char_received(info->tty,0xff,0xff);
		}
		break;
	}
	return 0;
}
#endif


/* Finally, routines used to initialize the serial driver. */

static struct tty_operations duart_ops = {
	.open = duart_open,
	.close = duart_close,
	.write = duart_write,
	.flush_chars = duart_flush_chars,
	.write_room = duart_write_room,
	.chars_in_buffer = duart_chars_in_buffer,
	.flush_buffer = duart_flush_buffer,
	.ioctl = duart_ioctl,
	.throttle = duart_throttle,
	.unthrottle = duart_unthrottle,
	.set_termios = duart_set_termios,
	.stop = duart_stop,
	.start = duart_start,
	.hangup = duart_hangup,
	.read_proc = duart_readproc,
	.wait_until_sent = duart_wait_until_sent,
 	.tiocmget = duart_tiocmget,
	.tiocmset = duart_tiocmset,
};

/* duart_init inits the driver */
static int __init duart_68692_init(void)
{
	int found=0;
	struct sag_serial	*info;
	unsigned long		flags;
	int			i;
	long 			addr,peek;

	/* Setup base handler, and timer table. */
	duart_serial_driver = alloc_tty_driver(NR_PORTS);
	if (!duart_serial_driver)
		return -ENOMEM;

	/* Initialize the tty_driver structure */
	duart_serial_driver->owner = THIS_MODULE;
	duart_serial_driver->name = "ttyS";
	duart_serial_driver->devfs_name = "ttys/";
	duart_serial_driver->driver_name = "duart_serial";
	duart_serial_driver->major = TTY_MAJOR;
	duart_serial_driver->minor_start = 67;
	duart_serial_driver->type = TTY_DRIVER_TYPE_SERIAL;
	duart_serial_driver->subtype = SERIAL_TYPE_NORMAL;
	duart_serial_driver->init_termios = tty_std_termios;

	duart_serial_driver->init_termios.c_cflag =
		B9600 | CS8 | CREAD | HUPCL | CLOCAL;
	duart_serial_driver->flags = TTY_DRIVER_REAL_RAW;

	tty_set_operations(duart_serial_driver, &duart_ops);

	if (tty_register_driver(duart_serial_driver)) {
		printk("duart: Couldn't register serial driver\n");
		put_tty_driver(duart_serial_driver);
		return(-EBUSY);
	}

	local_irq_save(flags);

	/* look for a valid option */
	for(i=0,info = duart_table;i<(NR_PORTS>>1);i++,info++)
	{
		addr=(long)info->addr;
		peek=peek_byte(addr+15);
		if(peek==-1)
		{
			/* no valid option */
			info++;			/* advance to second channel */
			continue;
		}
		if(peek!=255)
		{
			/* is a parallel i/o card */
			info++;			/* advance to second channel */
			continue;
		}
		found++;
		/* enable both uarts on this card */
		info->flags &= ~ASYNC_UART_OFFLINE;
		info++;				/* advance to second channel */
		info->flags &= ~ASYNC_UART_OFFLINE;
	}
	printk("Sartorius PR5510/04 driver with %d optioncard(s) installed\n",found);

	/*
	 *	Configure all the attached serial ports.
	 */
	for (i = 0, info = duart_table; (i < NR_PORTS); i++, info++)
	{
		if(info->flags&ASYNC_UART_OFFLINE) continue;
		info->magic = SERIAL_MAGIC;
		info->line = i;
		info->tty = 0;
		info->custom_divisor = 16;
		info->close_delay = 50;
		info->closing_wait = 3000;
		info->x_char = 0;
		info->event = 0;
		info->count = 0;
		info->blocked_open = 0;
		INIT_WORK(&info->tqueue, duart_offintr, info);
		INIT_WORK(&info->tqueue_hangup, do_serial_hangup, info);
		init_waitqueue_head(&info->open_wait);
		init_waitqueue_head(&info->close_wait);

		info->imr = 0;
		duart_setsignals(info, 0, 0);
		duart_irqinit(info);

		printk("ttyS%d at 0x%04x (irq = %d)\n", info->line+3,
			(unsigned int) info->chanaddr, info->irq);
#ifdef CONFIG_PM
     serial_pm[i] = pm_register(PM_SYS_DEV, PM_SYS_COM, serial_pm_callback);
	    if (serial_pm[i])
		    serial_pm[i]->data = info;
#endif
	}

	local_irq_restore(flags);
	return 0;
}

module_init(duart_68692_init);

/****************************************************************************/
