
/*
This file is part of CanFestival, a library implementing CanOpen Stack.

Copyright (C): Edouard TISSERANT and Francis DUPIN

See COPYING file for copyrights details.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/poll.h>
#include <time.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <asm/ioctls.h>
#include <termios.h>
#include <assert.h>

#include <errno.h>

// #define DEBUG_DBG_CONSOLE_ON 1
#define DEBUG_ERR_CONSOLE_ON 1


#include "drivers/char/can4linux/can4linux.h"

#include "can_driver.h"
#include "timer.h"
#include "timers_driver.h"

#define CANDRIVER_RECEIVE_TIMEOUT_MSEC	5000


#ifdef DEBUG_DBG_CONSOLE_ON
char dbgbuf[200];
static int dbgcount = 0;

char * dbghex (char *obuf, unsigned char* buf, unsigned char len)
{
	int i;

	sprintf (obuf, "%4.4d) ", dbgcount++);
	for (i=0; i<len; i++)
		sprintf (obuf+ 3*i + 6, "%02x ", buf[i]);

	return obuf;
}
#endif

/***************************************************************************/
int TranslateBaudeRate(char* baud_opt){
	if(!strcmp( baud_opt, "1M")) return CAN_BAUD_1M;
	if(!strcmp( baud_opt, "1000K")) return CAN_BAUD_1M;
	if(!strcmp( baud_opt, "500K")) return CAN_BAUD_500K;
	if(!strcmp( baud_opt, "250K")) return CAN_BAUD_250K;
	if(!strcmp( baud_opt, "125K")) return CAN_BAUD_125K;
	if(!strcmp( baud_opt, "100K")) return CAN_BAUD_100K;
	if(!strcmp( baud_opt, "50K")) return CAN_BAUD_50K;
	if(!strcmp( baud_opt, "20K")) return CAN_BAUD_20K;
	if(!strcmp( baud_opt, "10K")) return CAN_BAUD_10K;
	if(!strcmp( baud_opt, "5K")) return CAN_BAUD_5K;
	if(!strcmp( baud_opt, "none")) return 0;
	return 0x0000;
}


/*********functions which permit to communicate with the board****************/
UNS8 canReceive_driver(CAN_HANDLE fd0, Message *m)
{
  int res;
  int timeout_ms;
  canmsg_t canmsg;
  struct pollfd fdlist[1];
  // prepare poll call
  fdlist[0].fd = *((int*)fd0);
  fdlist[0].events = POLLIN | POLLPRI;

  canmsg.flags = 0; /* Ensure standard receive, not required for LinCAN>=0.3.1 */

  do{
#if 1
  	  TIMEVAL timeout;
	  timeout = getNextTimeDelta();
	  if (timeout == TIMEVAL_MAX)
		  timeout_ms = -1; // infinite waiting
	  else
		  timeout_ms = TIMEVAL_TO_MS(timeout);
#else
	timeout_ms = CANDRIVER_RECEIVE_TIMEOUT_MSEC;
#endif
	res = poll(fdlist, 1, timeout_ms);
	// if something available --> read it
	if (res > 0
	    &&
	    ( fdlist[0].revents & (POLLIN | POLLPRI) ) != 0 )
	{
		res = read(*((int*)fd0),&canmsg,1);
	}
	else
	{
		res = 0;
#if 1
		int err;

		if ((err = EnterMutex()) == 0)
		{
			TimeDispatch();
			LeaveMutex();
		}
		else
		{
			MSG_ERR(0, "timer_notify - Mutex deadlock (retry later): ", err);
		}
#endif
	}

	if((res<0)&&(errno == -EAGAIN))
		res = 0;
  }while(res==0);

  if(res != 1)		//sizeof(canmsg_t)) // No new message
  {
    MSG_ERR(0, "ERROR: Could not receive telegram!", res);

    return 1;
  }
  if(canmsg.flags&MSG_EXT){
    /* There is no mark for extended messages in CanFestival */;
  }

  MSG_DBG(canmsg.id, dbghex(dbgbuf, canmsg.data, 8), 0);

  m->cob_id.w = canmsg.id;
  m->len = canmsg.length;
  if(canmsg.flags&MSG_RTR){
    m->rtr = 1;
  }else{
    m->rtr = 0;
    memcpy(m->data,canmsg.data,8);
  }

  return 0;
}

/***************************************************************************/
UNS8 canSend_driver(CAN_HANDLE fd0, Message *m)
{
  int res;
  int retry;
  canmsg_t canmsg;
//  struct timespec timeout = {0,1000000};


  canmsg.flags = 0;
  canmsg.id = m->cob_id.w;
  canmsg.length = m->len;
  if(m->rtr){
    canmsg.flags |= MSG_RTR;
  }else{
    memcpy(canmsg.data,m->data,8);
  }

  if(canmsg.id >= 0x800){
    canmsg.flags |= MSG_EXT;
  }
#if 0
  // special error test for external DO
  if(canmsg.id == 0x63e){
	  MSG_ERR(0, "ERROR: Sending illegal 63E message", 0);
  }
#endif

  /* retry up to 5 times */
  for (retry = res = 0; res != 1 && retry < 200; retry++) {

  	res = write(*((int*)fd0),&canmsg,1);	// parameter 'count' defines the amount of can messages!  sizeof(canmsg_t));
	if (res == 0)
	{
		MSG_DBG(0, "ERROR: Send retry:", retry);
//		nanosleep(&timeout, NULL);
	}
  }

  MSG_DBG(canmsg.id, dbghex(dbgbuf, canmsg.data, 8), res);

  if(res!=1)	// sizeof(canmsg_t))
  {
	  /* try to reset the CAN controller */
//	  CanStatusPar_t status;
//	  MSG_ERR(0, "CANChip RESET executed", 0);
/*	  Command_par_t cmd;
	  cmd.cmd = CMD_STOP;
	  ioctl(*((int *) fd0), COMMAND, &cmd);

	  cmd.cmd = CMD_FLUSH;
	  ioctl(*((int *) fd0), COMMAND, &cmd);

	  cmd.cmd = CMD_RESET;
	  ioctl(*((int *) fd0), COMMAND, &cmd);

	  cmd.cmd = CMD_START;
	  ioctl(*((int *) fd0), COMMAND, &cmd);
*/
//	  ioctl(*((int *) fd0), STATUS, &status);
	  MSG_ERR(0, "ERROR: Could not send telegram!", retry);

	  return 1;
  }
  else
  {
	  return 0;
  }
}

/***************************************************************************/
static const char lnx_can_dev_prefix[] = "/dev/can";

CAN_HANDLE canOpen_driver(s_BOARD *board)
{
  int name_len = strlen(board->busname);
  int prefix_len = strlen(lnx_can_dev_prefix);
  char dev_name[prefix_len+name_len+1];
  int o_flags = 0;
  CAN_HANDLE fd0;
  int baudrate = 0;

  fd0=malloc(sizeof(int));
  if(fd0==NULL)
    return NULL;

  /*o_flags = O_NONBLOCK;*/

  memcpy(dev_name,lnx_can_dev_prefix,prefix_len);
  memcpy(dev_name+prefix_len,board->busname,name_len);
  dev_name[prefix_len+name_len] = 0;

  *((int *) fd0) = open(dev_name, O_RDWR|o_flags);
  if(*((int *) fd0) < 0){
    fprintf(stderr,"!!! Board %s is unknown. See can_4linux.c\n", board->busname);

    /* !!!!!!!! Error exit !!!!!!!!! */
    free(fd0);
    return NULL;

  }

  if (board->baudrate != NULL
      &&
      (baudrate = TranslateBaudeRate(board->baudrate)) > 0)
  {
	  // set the desired baudrate
	  Config_par_t  cfg;
	  cfg.target = CONF_TIMING;
	  cfg.val1   = baudrate;
	  ioctl(*((int *) fd0), CONFIG, &cfg);

  }
  else
  {
	  fprintf(stderr,"!!! Baudrate %s is unknown. See can_4linux.c\n", board->baudrate);

	  /* !!!!!!!! Error exit !!!!!!!!! */
	  close(*((int *) fd0));
	  free(fd0);
	  return NULL;

  }


  return fd0;

}

/***************************************************************************/
int canStatus_driver(CAN_HANDLE fd0)
{
    // Get the Status
    CanStatusPar_t  sts;
    ioctl(*((int *) fd0), CONFIG, &sts);
    return sts.status ;
}

/***************************************************************************/
int canClose_driver(CAN_HANDLE fd0)
{
  if(!fd0)
    return 0;
  close(*((int*)fd0));
  return 0;
}
