/*
  This file is part of CanFestival, a library implementing CanOpen
  Stack.

  Copyright (C): Edouard TISSERANT and Francis DUPIN

  See COPYING file for copyrights details.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
  USA
*/
/*!
** @file   objacces.c
** @author Edouard TISSERANT and Francis DUPIN
** @date   Tue Jun  5 08:55:23 2007
**
** @brief
**
**
*/




/* #define DEBUG_WAR_CONSOLE_ON */
/* #define DEBUG_ERR_CONSOLE_ON 1  */


#include "objacces.h"


/*!
**
**
** @param idx
** @param subIdx
** @param sizeDataDict
** @param sizeDataGiven
** @param code
**
** @return
**/
UNS8 accessDictionaryError(UNS16 idx, UNS8 subIdx,
                           UNS8 sizeDataDict, UNS8 sizeDataGiven, UNS32 code)
{
#ifdef DEBUG_WAR_CONSOLE_ON
  MSG_WAR(0x2B09,"Dictionary idx : ", idx);
  MSG_WAR(0X2B10,"           subidx : ", subIdx);
  switch (code) {
  case  OD_NO_SUCH_OBJECT:
    MSG_WAR(0x2B11,"Index not found ", idx);
    break;
  case OD_NO_SUCH_SUBINDEX :
    MSG_WAR(0x2B12,"SubIndex not found ", subIdx);
    break;
  case OD_WRITE_NOT_ALLOWED :
    MSG_WAR(0x2B13,"Write not allowed, data is read only ", idx);
    break;
  case OD_LENGTH_DATA_INVALID :
    MSG_WAR(0x2B14,"Conflict size data. Should be (bytes)  : ", sizeDataDict);
    MSG_WAR(0x2B15,"But you have given the size  : ", sizeDataGiven);
    break;
  case OD_NOT_MAPPABLE :
    MSG_WAR(0x2B16,"Not mappable data in a PDO at idx    : ", idx);
    break;
  case OD_VALUE_TOO_LOW :
    MSG_WAR(0x2B17,"Value range error : value too low. SDOabort : ", code);
    break;
  case OD_VALUE_TOO_HIGH :
    MSG_WAR(0x2B18,"Value range error : value too high. SDOabort : ", code);
    break;
  case OD_READ_NOT_ALLOWED :
    MSG_WAR(0x2B19,"Read not allowed, data is write only ", idx);
    break;
	  default :
    MSG_WAR(0x2B20, "Unknown error code : ", code);
  }
#endif
  return 0;
}

/*!
 **
 **
 ** @param d
 ** @param wIndex
 ** @param bSubidx
 ** @param pDestData
 ** @param pExpectedSize
 ** @param pDataType
 **
 ** @return
 **/
UNS32 getODentryPtr( CO_Data* d,
				   UNS16 wIndex,
				   UNS8 bSubidx,
				   void * * ppbData,
				   UNS8 * pSize,
				   UNS8 * pDataType)
{ /* DO NOT USE MSG_ERR because the macro may send a PDO -> infinite
	loop if it fails. */
	UNS32 errorCode;
	const indextable *ptrTable;
	ODCallback_t *Callback;

	ptrTable = (*d->scanIndexOD)(wIndex, &errorCode, &Callback);

	if (errorCode != OD_SUCCESSFUL)
		return errorCode;
	if( ptrTable->bSubCount <= bSubidx ) {
		/* Subidx not found */
		accessDictionaryError(wIndex, bSubidx, 0, 0, OD_NO_SUCH_SUBINDEX);
		return OD_NO_SUCH_SUBINDEX;
	}

	*pDataType = ptrTable->pSubindex[bSubidx].bDataType;
	*pSize = ptrTable->pSubindex[bSubidx].size;
	*ppbData = ptrTable->pSubindex[bSubidx].pObject;
    return OD_SUCCESSFUL;
}


/*!
**
**
** @param d
** @param wIndex
** @param bSubidx
** @param pDestData
** @param pExpectedSize
** @param pDataType
** @param checkAccess
** @param endianize
**
** @return
**/
UNS32 _getODentry( CO_Data* d,
                   UNS16 wIndex,
                   UNS8 bSubidx,
                   void * pDestData,
                   UNS8 * pExpectedSize,
                   UNS8 * pDataType,
                   UNS8 checkAccess,
                   UNS8 endianize)
{ /* DO NOT USE MSG_ERR because the macro may send a PDO -> infinite
    loop if it fails. */
  UNS32 errorCode;
  UNS8 szData;
  const indextable *ptrTable;
  ODCallback_t *Callback;

  ptrTable = (*d->scanIndexOD)(wIndex, &errorCode, &Callback);

  if (errorCode != OD_SUCCESSFUL)
    return errorCode;
  if( ptrTable->bSubCount <= bSubidx ) {
    /* Subidx not found */
    accessDictionaryError(wIndex, bSubidx, 0, 0, OD_NO_SUCH_SUBINDEX);
    return OD_NO_SUCH_SUBINDEX;
  }

  if (checkAccess && (ptrTable->pSubindex[bSubidx].bAccessType == WO)) {
    MSG_WAR(0x2B30, "Access Type : ", ptrTable->pSubindex[bSubidx].bAccessType);
    accessDictionaryError(wIndex, bSubidx, 0, 0, OD_READ_NOT_ALLOWED);
    return OD_READ_NOT_ALLOWED;
  }

  *pDataType = ptrTable->pSubindex[bSubidx].bDataType;
  szData = ptrTable->pSubindex[bSubidx].size;

  if(*pExpectedSize == 0 ||
     *pExpectedSize == szData ||
     *pDataType == visible_string ) {
    /* We allow to fetch a shorter string than expected */

	  if (*pExpectedSize > 0 && *pExpectedSize < szData)
		  szData = *pExpectedSize;

#  ifdef CANOPEN_BIG_ENDIAN
    if(endianize && *pDataType > boolean && *pDataType < visible_string) {
      /* data must be transmited with low byte first */
      UNS8 i, j = 0;
      MSG_WAR(boolean, "data type ", *pDataType);
      MSG_WAR(visible_string, "data type ", *pDataType);
      for ( i = szData ; i > 0 ; i--) {
        MSG_WAR(i," ", j);
        ((UNS8*)pDestData)[j++] =
          ((UNS8*)ptrTable->pSubindex[bSubidx].pObject)[i-1];
      }
    }
    else /* It it is a visible string no endianisation to perform */
      memcpy(pDestData, ptrTable->pSubindex[bSubidx].pObject,szData);
#  else
    memcpy(pDestData, ptrTable->pSubindex[bSubidx].pObject,szData);
#  endif

    *pExpectedSize = szData;
#if 0
    /* Me laisser a, please ! (FD) */
    {
      UNS8 i;
      for (i = 0 ; i < 10 ; i++) {
        MSG_WAR(*pExpectedSize, "dic data= ",
                *(UNS8 *)(ptrTable->pSubindex[bSubidx].pObject + i));
      }

    }
#endif
    return OD_SUCCESSFUL;
  }
  else { /* Error ! */
    *pExpectedSize = szData;
    accessDictionaryError(wIndex, bSubidx, szData,
                          *pExpectedSize, OD_LENGTH_DATA_INVALID);
    return OD_LENGTH_DATA_INVALID;
  }
}

/*!
**
**
** @param d
** @param wIndex
** @param bSubidx
** @param pDestData
** @param pExpectedSize
** @param pDataType
** @param checkAccess
**
** @return
**/
UNS32 getODentry( CO_Data* d,
                  UNS16 wIndex,
                  UNS8 bSubidx,
                  void * pDestData,
                  UNS8 * pExpectedSize,
                  UNS8 * pDataType,
                  UNS8 checkAccess)
{
  return _getODentry( d,
                      wIndex,
                      bSubidx,
                      pDestData,
                      pExpectedSize,
                      pDataType,
                      checkAccess,
                      1);//endianize
}

/*!
**
**
** @param d
** @param wIndex
** @param bSubidx
** @param pDestData
** @param pExpectedSize
** @param pDataType
** @param checkAccess
**
** @return
**/
UNS32 readLocalDict( CO_Data* d,
                     UNS16 wIndex,
                     UNS8 bSubidx,
                     void * pDestData,
                     UNS8 * pExpectedSize,
                     UNS8 * pDataType,
                     UNS8 checkAccess)
{
  return _getODentry( d,
                      wIndex,
                      bSubidx,
                      pDestData,
                      pExpectedSize,
                      pDataType,
                      checkAccess,
                      0);//do not endianize
}

/*!
**
**
** @param d
** @param wIndex
** @param bSubidx
** @param pSourceData
** @param pExpectedSize
** @param checkAccess
** @param endianize
**
** @return
**/
UNS32 _setODentry( CO_Data* d,
                   UNS16 wIndex,
                   UNS8 bSubidx,
                   void * pSourceData,
                   UNS8 * pExpectedSize,
                   UNS8 checkAccess,
                   UNS8 endianize)
{
  UNS8 szData;
  UNS8 dataType;
  UNS8 actLen;
  UNS32 errorCode;
  const indextable *ptrTable;
  ODCallback_t *Callback;

  ptrTable =(*d->scanIndexOD)(wIndex, &errorCode, &Callback);
  if (errorCode != OD_SUCCESSFUL)
    return errorCode;

  if( ptrTable->bSubCount <= bSubidx ) {
    /* Subidx not found */
    accessDictionaryError(wIndex, bSubidx, 0, *pExpectedSize, OD_NO_SUCH_SUBINDEX);
    return OD_NO_SUCH_SUBINDEX;
  }
  if (checkAccess && (ptrTable->pSubindex[bSubidx].bAccessType == RO)) {
    MSG_WAR(0x2B25, "Access Type : ", ptrTable->pSubindex[bSubidx].bAccessType);
    accessDictionaryError(wIndex, bSubidx, 0, *pExpectedSize, OD_WRITE_NOT_ALLOWED);
    return OD_WRITE_NOT_ALLOWED;
  }


  dataType = ptrTable->pSubindex[bSubidx].bDataType;
  szData = ptrTable->pSubindex[bSubidx].size;
  actLen = *pExpectedSize == 0 ? szData : *pExpectedSize;

  if( *pExpectedSize == 0 ||
      *pExpectedSize == szData ||
      (dataType == visible_string && *pExpectedSize < szData)) /* We
                                                                  allow to store a shorter string than entry size */
    {
#ifdef CANOPEN_BIG_ENDIAN
      if(endianize && dataType > boolean && dataType < visible_string)
        {
          /* we invert the data source directly. This let us do range
            testing without */
          /* additional temp variable */
          UNS8 i;
          for ( i = 0 ; i < ( ptrTable->pSubindex[bSubidx].size >> 1)  ; i++)
            {
              UNS8 tmp =((UNS8 *)pSourceData) [(ptrTable->pSubindex[bSubidx].size - 1) - i];
              ((UNS8 *)pSourceData) [(ptrTable->pSubindex[bSubidx].size - 1) - i] = ((UNS8 *)pSourceData)[i];
              ((UNS8 *)pSourceData)[i] = tmp;
            }
        }
#endif
      errorCode = (*d->valueRangeTest)(dataType, pSourceData);
      if (errorCode) {
        accessDictionaryError(wIndex, bSubidx, szData, *pExpectedSize, errorCode);
        return errorCode;
      }
      memcpy(ptrTable->pSubindex[bSubidx].pObject,pSourceData, *pExpectedSize);
      *pExpectedSize = szData;

      /* Callbacks */
      if(Callback && Callback[bSubidx]){
		  (*Callback[bSubidx])(d, ptrTable, bSubidx, CBTRIGG_ODWRITTEN, actLen);
      }

      /* TODO : Store dans NVRAM */
      if (ptrTable->pSubindex[bSubidx].bAccessType & TO_BE_SAVE){
        (*d->storeODSubIndex)(wIndex, bSubidx);
      }
      return OD_SUCCESSFUL;
    }else{
      *pExpectedSize = szData;
      accessDictionaryError(wIndex, bSubidx, szData, *pExpectedSize, OD_LENGTH_DATA_INVALID);
      return OD_LENGTH_DATA_INVALID;
    }
}

/*!
**
**
** @param d
** @param wIndex
** @param bSubidx
** @param pSourceData
** @param pExpectedSize
** @param checkAccess
**
** @return
**/
UNS32 setODentry( CO_Data* d,
                  UNS16 wIndex,
                  UNS8 bSubidx,
                  void * pSourceData,
                  UNS8 * pExpectedSize,
                  UNS8 checkAccess)
{
  return _setODentry( d,
                      wIndex,
                      bSubidx,
                      pSourceData,
                      pExpectedSize,
                      checkAccess,
                      1);//endianize
}


/*!
 **
 **
 ** @param d
 ** @param wIndex
 ** @param bSubidx
 ** @param ExpectedSize
 ** @param checkAccess
 **
 ** @return
 **/
UNS32 checkODentry( CO_Data* d,
				   UNS16 wIndex,
				   UNS8 bSubidx,
				   UNS32 ExpectedSize,
				   UNS8 checkAccess)
{
	UNS32 szData;
	UNS32 actLen;
	UNS32 errorCode;
	const indextable *ptrTable;
	ODCallback_t *Callback;

	ptrTable =(*d->scanIndexOD)(wIndex, &errorCode, &Callback);
	if (errorCode != OD_SUCCESSFUL)
		return errorCode;

	if( ptrTable->bSubCount <= bSubidx ) {
		/* Subidx not found */
		accessDictionaryError(wIndex, bSubidx, 0, ExpectedSize, OD_NO_SUCH_SUBINDEX);
		return OD_NO_SUCH_SUBINDEX;
	}
	if (checkAccess && (ptrTable->pSubindex[bSubidx].bAccessType == RO)) {
		MSG_WAR(0x2B25, "Access Type : ", ptrTable->pSubindex[bSubidx].bAccessType);
		accessDictionaryError(wIndex, bSubidx, 0, ExpectedSize, OD_WRITE_NOT_ALLOWED);
		return OD_WRITE_NOT_ALLOWED;
	}


	szData = ptrTable->pSubindex[bSubidx].size;
	actLen = ExpectedSize == 0 ? szData : ExpectedSize;
	MSG_ERR(ExpectedSize, "expected<< check >>> actual", actLen);

	if (actLen > szData) {
		accessDictionaryError(wIndex, bSubidx, szData, ExpectedSize, OD_LENGTH_DATA_INVALID);
		return OD_LENGTH_DATA_INVALID;
	}

	/* Callbacks */
	if(Callback && Callback[bSubidx]){
		return (*Callback[bSubidx])(d, ptrTable, bSubidx, CBTRIGG_ODCHECK, actLen);
	}
	else
		return OD_SUCCESSFUL;
}


/*!
**
**
** @param d
** @param wIndex
** @param bSubidx
** @param pSourceData
** @param pExpectedSize
** @param checkAccess
**
** @return
**/
UNS32 writeLocalDict( CO_Data* d,
                      UNS16 wIndex,
                      UNS8 bSubidx,
                      void * pSourceData,
                      UNS8 * pExpectedSize,
                      UNS8 checkAccess)
{
  return _setODentry( d,
                      wIndex,
                      bSubidx,
                      pSourceData,
                      pExpectedSize,
                      checkAccess,
                      0);//do not endianize
}

/*!
**
**
** @param d
** @param wIndex
** @param errorCode
** @param Callback
**
** @return
**/
const indextable * scanIndexOD (CO_Data* d, UNS16 wIndex, UNS32 *errorCode, ODCallback_t **Callback)
{
  return (*d->scanIndexOD)(wIndex, errorCode, Callback);
}

/*!
**
**
** @param d
** @param wIndex
** @param bSubidx
** @param Callback
**
** @return
**/
UNS32 RegisterSetODentryCallBack(CO_Data* d, UNS16 wIndex, UNS8 bSubidx, ODCallback_t Callback)
{
  UNS32 errorCode;
  ODCallback_t *CallbackList;

  scanIndexOD (d, wIndex, &errorCode, &CallbackList);
  if(errorCode == OD_SUCCESSFUL && CallbackList)
    CallbackList[bSubidx] = Callback;
  return errorCode;
}

/*!
**
**
** @param wIndex
** @param bSubidx
**/
void _storeODSubIndex (UNS16 wIndex, UNS8 bSubidx){}

/*!
 ** Initialize the object dictionary
 **
 ** @param d Object dictionary to initOD
 **
 ** @return
 **/

UNS32 initOD( CO_Data* d)
{
	int i;


	/* SDO */
	for (i=0; i < SDO_MAX_SIMULTANEOUS_TRANSFERTS; i++)
	{
		d->transfers[i] = s_transfer_Initializer;
	}

	/* State machine */
	d->nodeState = Unknown_state;
	memset (&(d->CurrentCommunicationState), sizeof(d->CurrentCommunicationState), 0);
	d->CurrentCommunicationState = s_state_communication_Initializer;
	d->initialisation = _initialisation;     /* initialisation */
	d->preOperational = _preOperational;     /* preOperational */
	d->operational = _operational;           /* operational */
	d->stopped = _stopped;                   /* stopped */

	/* NMT-heartbeat */
	d->ProducerHeartBeatTimer = TIMER_NONE; /* ProducerHeartBeatTimer */
	d->heartbeatError = _heartbeatError;    /* heartbeatError */


	for (i=0; i < NMT_MAX_NODE_ID; i++)
	{
		d->NMTable[i] = NMTable_Initializer;
	}
	/* SYNC */
	d->syncTimer = TIMER_NONE;
	d->post_sync = _post_sync;
	d->post_TPDO = _post_TPDO;

	/* General */
	d->toggle = 0;
	d->canHandle = NULL;
	d->storeODSubIndex = _storeODSubIndex;

	/* DCF concise */
	d->dcf_cursor = NULL;
	d->dcf_count_targets = 1;

	/* EMCY */
	d->error_state = Error_free;
	for (i=0; i < EMCY_MAX_ERRORS; i++)
	{
		d->error_data[EMCY_MAX_ERRORS] = ERROR_DATA_INITIALIZER;
	}

	d->post_emcy = _post_emcy;

	/* RAW */
	for (i=0; i < MAXRAWCHANNELS; i++)
	{
		d->RAWCallbackTable[i].cobId = 0;
		d->RAWCallbackTable[i].Callback = NULL;
		d->RAWCallbackTable[i].Callbackpara = NULL;
	}

	return OD_SUCCESSFUL;
}

