/*
This file is part of CanFestival, a library implementing CanOpen Stack.

Copyright (C): Edouard TISSERANT and Francis DUPIN

See COPYING file for copyrights details.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/*!
** @file   sdo.c
** @author Edouard TISSERANT and Francis DUPIN
** @date   Tue Jun  5 09:32:32 2007
**
** @brief
**
**
*/

#define USE_BLOCK_TRANSFER 0

//#define DEBUG_WAR_CONSOLE_ON 1
#define DEBUG_ERR_CONSOLE_ON  1
//#define DEBUG_DBG_CONSOLE_ON 1
#define DEBUG_LAST_MESSAGE 1

#include "objacces.h"
#include "sdo.h"
#include "canfestival.h"

/* Uncomment if your compiler does not support inline functions */
#define NO_INLINE

#ifdef NO_INLINE
  #define INLINE
#else
  #define INLINE inline
#endif

#define min(a,b) a<b?a:b

int sdoBlocked = 0;

#ifdef DEBUG_LAST_MESSAGE

static char dbgbuf[200];

char * dbgmsg (char *obuf, const char *text, struct timeval *t, Message *m)
{
	sprintf (obuf, "%6.6d,%6.6d %s  %3.3X  %2.2X %2.2X %2.2X %2.2X %2.2X %2.2X %2.2X %2.2X ",
                (int) t->tv_sec, (int) t->tv_usec, text, m->cob_id.w,
                m->data[0], m->data[1], m->data[2], m->data[3],
                m->data[4], m->data[5], m->data[6], m->data[7]  );

	return obuf;
}
#endif

/*Internals prototypes*/

/*!
** Called by writeNetworkDict
**
** @param d
** @param nodeId
** @param idx
** @param subIdx
** @param count
** @param dataType
** @param data
** @param Callback
** @param endianize
**
** @return
**/
INLINE UNS8 _writeNetworkDict (CO_Data* d, UNS8 nodeId, UNS16 idx,
		       UNS8 subIdx, UNS8 count, UNS8 dataType, void *data, SDOCallback_t Callback, void * Callbackpara, UNS8 endianize);

/*!
** Called by readNetworkDict
**
** @param d
** @param nodeId
** @param idx
** @param subIdx
** @param dataType
** @param Callback
**
** @return
**/
INLINE UNS8 _readNetworkDict (CO_Data* d, UNS8 nodeId, UNS16 idx, UNS8 subIdx,
	UNS8 dataType, SDOCallback_t Callback, void * Callbackpara);

/*! Terminate communication line (and evtl. call the callback function)
 **
 **
 ** @param d
 ** @param id
 **/

void terminateSDOline(CO_Data* d, UNS8 lineno)
{
	resetSDOline(d, lineno);
	if (d->transfers[lineno].Callback)
		(*d->transfers[lineno].Callback)(d,lineno,d->transfers[lineno].nodeId,&(d->transfers[lineno]));
}

/*!
**
**
** @param d
** @param id
**/
void SDOTimeoutAlarm(CO_Data* d, UNS32 id)
{
	sdoBlocked = 1;

    MSG_ERR(0x1A01, "SDO timeout. SDO response not received.", 0);
	MSG_ERR(d->transfers[id].cobidRx, "server node : ", d->transfers[id].nodeId);
    MSG_ERR(0x2A02, "      idx : ", d->transfers[id].idx);
    MSG_ERR(0x2A02, "   subIdx : ", d->transfers[id].subIdx);
    MSG_ERR(0,dbgmsg(dbgbuf, "LastOUT:", &LastTxTime, &LastTxMsg), 0);
    MSG_ERR(0,dbgmsg(dbgbuf, "LastIN :", &LastRxTime, &LastRxMsg), 0);

    /* Reset timer handler */
    d->transfers[id].timer = TIMER_NONE;
    /*Set aborted state*/
    d->transfers[id].state = SDO_ABORTED_INTERNAL;
    /* Sending a SDO abort */
	//TODO: ????? is that ok???
    sendSDOabort(d, d->transfers[id].cobidTx,
		 d->transfers[id].idx, d->transfers[id].subIdx, SDOABT_TIMED_OUT);
    d->transfers[id].abortCode = SDOABT_TIMED_OUT;
    /* Call the user function to inform of the problem.*/

#if 0
	if(d->transfers[id].Callback)
    	/*If ther is a callback, it is responsible to close SDO transfer (client)*/
    	(*d->transfers[id].Callback)(d,d->transfers[id].nodeId);
    else if(d->transfers[id].whoami == SDO_SERVER)
    	/*Else, if server, reset the line*/
    	resetSDOline(d, (UNS8)id);
#endif
   terminateSDOline(d, (UNS8) id);

	if (d->SDOTimeOutCallback)
		(*d->SDOTimeOutCallback)(d,id);


}

/*!
 **
 **
 ** @param d
 ** @param Callback		Callback function to be registered for Timeout
 **/
void RegisterSDOTimeOutCallBack(CO_Data* d,  SDOTimeOutCallback_t Callback)
{
	d->SDOTimeOutCallback = Callback;
}

#define StopSDO_TIMER(id) \
MSG_WAR(0x3A05, "StopSDO_TIMER for line : ", line);\
d->transfers[id].timer = DelAlarm(d->transfers[id].timer);

#define StartSDO_TIMER(id) \
MSG_WAR(0x3A06, "StartSDO_TIMER for line : ", line);\
d->transfers[id].timer = SetAlarm(d,id,&SDOTimeoutAlarm,MS_TO_TIMEVAL(SDO_TIMEOUT_MS),0);

#define RestartSDO_TIMER(id) \
MSG_WAR(0x3A07, "restartSDO_TIMER for line : ", line);\
if(d->transfers[id].timer != TIMER_NONE) { StopSDO_TIMER(id) StartSDO_TIMER(id) }

/*!
** Reset all sdo buffers
**
** @param d
**/
void resetSDO (CO_Data* d)
{
  UNS8 j;

  /* transfer structure initialization */
    for (j = 0 ; j < SDO_MAX_SIMULTANEOUS_TRANSFERTS ; j++)
      resetSDOline(d, j);
}

/*!
**
**
** @param d
** @param line
**
** @return
**/
UNS32 SDOlineToObjdict (CO_Data* d, UNS8 line)
{
  UNS8      size;
  UNS32 errorCode;
  MSG_WAR(0x3A08, "Enter in SDOlineToObjdict ", line);
  size = (UNS8)d->transfers[line].offset;
  MSG_WAR(0xfe, "size = (UNS8)d->transfers[line].count ", (UNS8)d->transfers[line].count);
  MSG_WAR(0xfe, "size = (UNS8)d->transfers[line].offset ", (UNS8)d->transfers[line].offset);
  d->line_ptr = &(d->transfers[line]);
  errorCode = setODentry(d, d->transfers[line].idx, d->transfers[line].subIdx,
			 (void *) d->transfers[line].data, &size, 1);
  if (errorCode != OD_SUCCESSFUL)
    return errorCode;
  MSG_WAR(0x3A08, "exit of SDOlineToObjdict ", line);
  return 0;

}

/*!
**
**
** @param d
** @param line
**
** @return
**/
UNS32 objdictToSDOline (CO_Data* d, UNS8 line)
{
  UNS8  size = 0;
  UNS8  dataType;
  UNS32 errorCode;

  MSG_WAR(0x3A05, "objdict->line idx : ", d->transfers[line].idx);
  MSG_WAR(0x3A06, "  subIdx : ", d->transfers[line].subIdx);

  errorCode = getODentry(d, 	d->transfers[line].idx,
  				d->transfers[line].subIdx,
  				(void *)d->transfers[line].data,
  				&size, &dataType, 0);

  if (errorCode != OD_SUCCESSFUL)
    return errorCode;

  d->transfers[line].count = size;
  d->transfers[line].offset = 0;
#if 0
   /*Me laisser a, please ! (FD)*/
  {
    UNS8 i;
    for (i = 0 ; i < 10 ; i++) {
      MSG_WAR(i, "data= ", d->transfers[line].data[i]);
    }
  }
#endif
  return 0;
}

/*!
**
**
** @param d
** @param line
** @param nbBytes
** @param data
**
** @return
**/
UNS8 lineToSDO (CO_Data* d, UNS8 line, UNS8 nbBytes, UNS8* data) {
  UNS8 i;
  UNS8 offset;

  if ((d->transfers[line].offset + nbBytes) > SDO_MAX_LENGTH_TRANSFERT) {
    MSG_ERR(0x1A10,"SDO Size of data too large. Exceed SDO_MAX_LENGTH_TRANSFERT", nbBytes);
    return 0xFF;
  }
    if ((d->transfers[line].offset + nbBytes) > d->transfers[line].count) {
    MSG_ERR(0x1A11,"SDO Size of data too large. Exceed count", nbBytes);
    return 0xFF;
  }
  offset = (UNS8)d->transfers[line].offset;
  for (i = 0 ; i < nbBytes ; i++)
    * (data + i) = d->transfers[line].data[offset + i];
  d->transfers[line].offset = d->transfers[line].offset + nbBytes;
  return 0;
}

/*!
**
**
** @param lineptr
** @param nbBytes
** @param data
**
** @return
**/
UNS8 SDOtoLine (s_transfer *lineptr, UNS8 nbBytes, UNS8* data)
{
  UNS8 i;
  UNS8 offset;

  if ((lineptr->offset + nbBytes) > SDO_MAX_LENGTH_TRANSFERT) {
    MSG_ERR(0x1A15,"SDO Size of data too large. Exceed SDO_MAX_LENGTH_TRANSFERT", nbBytes);
    return 0xFF;
  }
  offset = (UNS8)lineptr->offset;
  for (i = 0 ; i < nbBytes ; i++)
	  lineptr->data[offset + i] = * (data + i);
  lineptr->offset =lineptr->offset + nbBytes;
  return 0;
}

/*!
**
**
** @param d
** @param nodeId
** @param whoami
** @param idx
** @param subIdx
** @param abortCode
**
** @return
**/
UNS8 failedSDO (CO_Data* d, UNS8 nodeId, UNS8 whoami, UNS16 idx,
		UNS8 subIdx, UNS32 abortCode)
{
  UNS8 err;
  UNS8 line;
  s_transfer * lineptr;
  UNS32 cobidTx=0;
  UNS32 cobidRx=0;
  UNS16 nodeIdidx;

  err = getSDOlineOnUse( d, nodeId, whoami, &line, &lineptr );
  if (!err) /* If a line on use have been found.*/
  {
    MSG_WAR(0x3A20, "FailedSDO : line found : ", line);
	cobidTx = lineptr->cobidTx;
  }
  else
  {
	  getSDOcobid(d, nodeId, whoami, &cobidTx, &cobidRx, &nodeIdidx);
	  line = 0xff;
  }
  if ((! err) && (whoami == SDO_SERVER)) {
    resetSDOline( d, line );
    MSG_WAR(0x3A21, "FailedSDO : line released : ", line);
  }
  if ((! err) && (whoami == SDO_CLIENT)) {
    StopSDO_TIMER(line);
    d->transfers[line].state = SDO_ABORTED_INTERNAL;
//	d->transfers[line].abortCode = abortCode;
  }
  MSG_WAR(0x3A22, "Sending SDO abort ", 0);
  if (cobidTx != 0)
  	err = sendSDOabort(d, cobidTx, idx, subIdx, abortCode);

  if (err || cobidTx == 0) {
    MSG_WAR(0x3A23, "Unable to send the SDO abort", 0);
    err = 0xFF;
  }

  if (line != 0xff)
	  terminateSDOline(d, line);

  return err;
}

/*!
**
**
** @param d
** @param line
**/

void resetSDOline ( CO_Data* d, UNS8 line )
{
	MSG_WAR(0x3A25, "release SDO line nb : ", line);

	StopSDO_TIMER(line);
	d->transfers[line].inUse = 0;
}

/*!
**
**
** @param d
** @param line
** @param nodeId
** @param idx
** @param subIdx
** @param state
**
** @return
**/
UNS8 initSDOline (CO_Data* d, UNS8 line, UNS8 nodeId, UNS16 idx, UNS8 subIdx, UNS8 state)
{
	s_transfer * line_ptr = &(d->transfers[line]);
	MSG_WAR(0x3A25, "init SDO line nb : ", line);
	if (state == SDO_RESET)
	{
		d->transfers[line].inUse = 0;
		StopSDO_TIMER(line)
	}else
	{
		d->transfers[line].inUse = 1;
		StartSDO_TIMER(line)
	}
	line_ptr->nodeId = nodeId;
	line_ptr->idx = idx;
	line_ptr->subIdx = subIdx;
	line_ptr->state = state;

	line_ptr->toggle = 0;
	line_ptr->count = 0;
	line_ptr->offset = 0;
	line_ptr->dataType = 0;
	line_ptr->Callback = NULL;
	line_ptr->nodeIdidx = 0;
	line_ptr->cobidTx = 0;
	line_ptr->cobidRx = 0;
	line_ptr->abortCode = 0;

  	return 0;
}

/*!
**
**
** @param d
** @param whoami
** @param line
** @param linptr
 **
** @return
**/
UNS8 getSDOfreeLine ( CO_Data* d, UNS8 whoami, UNS8 *line, s_transfer ** lineptr )
{

  UNS8 i;

  for (i = 0 ; i < SDO_MAX_SIMULTANEOUS_TRANSFERTS ; i++){
    if ( ! d->transfers[i].inUse ) {
      *line = i;
	  *lineptr = &(d->transfers[i]);
	  d->transfers[i].whoami = whoami;
      return 0;
    } /* end if */
  } /* end for */
  MSG_ERR(0x1A25, "Too many SDO in progress. Aborted.", i);
  *lineptr = 0;
  return 0xFF;
}

/*!
**
**
** @param d
** @param nodeId
** @param whoami
** @param line
** @param linptr
**
** @return
**/
UNS8 getSDOlineOnUse (CO_Data* d, UNS8 nodeId, UNS8 whoami, UNS8 *line, s_transfer ** lineptr)
{

  UNS8 i;

  for (i = 0 ; i < SDO_MAX_SIMULTANEOUS_TRANSFERTS ; i++){
    if ( (d->transfers[i].inUse) &&
	 (d->transfers[i].nodeId == nodeId) &&
		  (d->transfers[i].whoami == whoami || whoami == SDO_UNSPECIFIED) ) {
      *line = i;
	  *lineptr = &(d->transfers[i]);
      return 0;
    }
  }
  *lineptr = 0;
  *line = 0xFF;
  return 0xFF;
}

/*!
 **
 **
 ** @param d
 ** @param nodeId
 ** @param whoami
 ** @param line
 ** @param linptr
 **
 ** @return
 **/
UNS8 getSDOlineClosed (CO_Data* d, UNS8 nodeId, UNS8 whoami, UNS8 *line, s_transfer ** lineptr)
{

	UNS8 i;

	for (i = 0 ; i < SDO_MAX_SIMULTANEOUS_TRANSFERTS ; i++){
		if ( ( ! d->transfers[i].inUse) &&
					(d->transfers[i].nodeId == nodeId) &&
					(d->transfers[i].whoami == whoami || whoami == SDO_UNSPECIFIED) ) {
			*line = i;
			*lineptr = &(d->transfers[i]);
			return 0;
					}
	}
	*lineptr = 0;
	*line = 0xFF;
	return 0xFF;
}

/*!
**
**
** @param d
** @param nodeId
** @param whoami
**
** @return
**/
UNS8 closeSDOtransfer (CO_Data* d, UNS8 nodeId, UNS8 whoami)
{
  UNS8 err;
  UNS8 line;
  s_transfer * lineptr;
  err = getSDOlineOnUse(d, nodeId, whoami, &line, &lineptr);
  if (err) {
    MSG_WAR(0x2A30, "No SDO communication to close for node : ", nodeId);
    return 0xFF;
  }
  resetSDOline(d, line);
  return 0;
}

/*!
 **
 **
 ** @param d
 ** @param nodeId
 ** @param whoami
 ** @param cobidTx
 ** @param cobidRx
 **
 ** @return
 **/

UNS8  getSDOcobid(CO_Data* d, UNS8 nodeId, UNS8 whoami,  UNS32 * cobidTx, UNS32 * cobidRx, UNS16 * nodeIdidx)
{

	UNS8 err = 0xFF;
	*cobidTx = 0;
	*cobidRx = 0;
	UNS16 offset;
	UNS16 lastIndex;
	int j;

	if (  whoami == SDO_SERVER )
	{
		/* Server COBID */
		offset = d->firstIndex->SDO_SVR;
		lastIndex = d->lastIndex->SDO_SVR;
		j = 0;
		if(offset) while (offset <= lastIndex)
		{
			if (	(j == 0 && d->objdict[offset].bSubCount <= 2)
					 ||	 ( j > 0 && d->objdict[offset].bSubCount <= 3))
			{
				return 0xFF;
			}
			if ((j == 0 && nodeId == *(d->bDeviceNodeId))
						||  (j > 0 && nodeId == *( (UNS8*) d->objdict[offset].pSubindex[3].pObject))
			   )
			{
				*cobidRx = *( (UNS32*) d->objdict[offset].pSubindex[1].pObject);
				*cobidTx = *( (UNS32*) d->objdict[offset].pSubindex[2].pObject);
				*nodeIdidx = j;
				return 0;
			}

			j++;
			offset++;
		}

	}
	else
	{
		/* Client CobID */
		offset = d->firstIndex->SDO_CLT;
		lastIndex = d->lastIndex->SDO_CLT;
		j = 0;
		if(offset) while (offset <= lastIndex)
		{
			if (d->objdict[offset].bSubCount <= 3)
			{
				return 0xFF;
			}
			if (nodeId == *( (UNS8*) d->objdict[offset].pSubindex[3].pObject))
			{
				*cobidTx = *( (UNS32*) d->objdict[offset].pSubindex[1].pObject);
				*cobidRx = *( (UNS32*) d->objdict[offset].pSubindex[2].pObject);
				*nodeIdidx = j;
				return 0;
			}

			j++;
			offset++;
		}


	}
	return err;
}

/*!
**
**
** @param d
** @param line
** @param nbBytes
**
** @return
**/
UNS8 getSDOlineRestBytes (CO_Data* d, UNS8 line, UNS8 * nbBytes)
{
  if (d->transfers[line].count == 0) /* if received initiate SDO protocol with e=0 and s=0 */
    * nbBytes = 0;
  else
    * nbBytes = (UNS8)d->transfers[line].count - (UNS8)d->transfers[line].offset;
  return 0;
}

/*!
**
**
** @param lineptr
** @param nbBytes
**
** @return
**/
UNS8 setSDOlineRestBytes (s_transfer *lineptr, UNS8 nbBytes)
{
  if (nbBytes > SDO_MAX_LENGTH_TRANSFERT) {
    MSG_ERR(0x1A35,"SDO Size of data too large. Exceed SDO_MAX_LENGTH_TRANSFERT", nbBytes);
    return 0xFF;
  }
  lineptr->count = nbBytes;
  return 0;
}

#if 0
/*!
**
**
** @param d
** @param whoami
** @param sdo
**
** @return
**/
UNS8 sendSDO (CO_Data* d, UNS8 whoami, s_SDO sdo)
{
	UNS16 offset;
	UNS16 lastIndex;
	UNS8 found = 0;
	Message m;
	UNS8 i;
	UNS32 * pwCobId = NULL;
	UNS8 * pwNodeId = NULL;

	MSG_WAR(0x3A38, "sendSDO",0);
	if( !((d->nodeState == Operational) ||  (d->nodeState == Pre_operational )))
	{
		MSG_WAR(0x2A39, "unable to send the SDO (not in op or pre-op mode", d->nodeState);
		return 0xFF;
	}

	/*get the server->client cobid*/
	if ( whoami == SDO_SERVER )
	{
		/*case server. Easy because today only one server SDO is authorized in CanFestival*/
		offset = d->firstIndex->SDO_SVR;
		if (offset == 0)
		{
			MSG_ERR(0x1A42, "SendSDO : No SDO server found", 0);
			return 0xFF;
		}
		//pwCobId = (UNS32*) d->objdict[offset].pSubindex[2].pObject;
		pwCobId = (UNS32*) d->objdict[offset + sdo.nodeId].pSubindex[2].pObject;

		MSG_WAR(0x3A41, "I am server. cobId : ", *pwCobId);
	}
	else
	{
		/*case client*/
		/* Get the client->server cobid.*/
		UNS16 sdoNum = 0;
		offset = d->firstIndex->SDO_CLT;
		lastIndex = d->lastIndex->SDO_CLT;
		if (offset == 0)
		{
			MSG_ERR(0x1A42, "SendSDO : No SDO client idx found", 0);
			return 0xFF;
		}
		/* First, have to find at the idx where is defined the communication with the server node */
		while (offset <= lastIndex)
		{
			MSG_WAR(0x3A43,"Reading idx : ", 0x1280 + sdoNum);
			if (d->objdict[offset].bSubCount <= 3) {
				MSG_ERR(0x1A28, "Subindex 3  not found at idx ", 0x1280 + sdoNum);
				return 0xFF;
			}
			pwNodeId = (UNS8*) d->objdict[offset].pSubindex[3].pObject;
			MSG_WAR(0x3A44, "Found nodeId server = ", *pwNodeId);
			if(*pwNodeId == sdo.nodeId)
			{
				found = 1;
				break;
			}
			offset ++;
			sdoNum ++;
		}
		if (! found)
		{
			MSG_WAR (0x2A45, "No SDO client corresponds to the mesage to send to node ", sdo.nodeId);
			return 0xFF;
		}
		/* Second, read the cobid client->server */
		pwCobId = (UNS32*) d->objdict[offset].pSubindex[1].pObject;
	}
	/* message copy for sending */
	m.cob_id.w = *pwCobId;
	m.rtr = NOT_A_REQUEST;
	/* the length of SDO must be 8 */
	m.len = 8;
	for (i = 0 ; i < 8 ; i++)
	{
		m.data[i] =  sdo.body.data[i];
	}
	return canSend(d->canHandle,&m);
}
#endif

/*!
**
**
** @param d
** @param whoami
** @param idx
** @param subIdx
** @param abortCode
**
** @return
**/
UNS8 sendSDOabort (CO_Data* d, UNS32 cobidTx, UNS16 idx, UNS8 subIdx, UNS32 abortCode)
{
  s_SDO sdo;
  UNS8 ret;
  MSG_WAR(0x2A50,"Sending SDO abort ", abortCode);
  sdo.cobid = cobidTx;
  sdo.body.data[0] = 0x80;
  /* Index */
  sdo.body.data[1] = idx & 0xFF; /* LSB */
  sdo.body.data[2] = (idx >> 8) & 0xFF; /* MSB */
  /* Subindex */
  sdo.body.data[3] = subIdx;
  /* Data */
  sdo.body.data[4] = (UNS8)(abortCode & 0xFF);
  sdo.body.data[5] = (UNS8)((abortCode >> 8) & 0xFF);
  sdo.body.data[6] = (UNS8)((abortCode >> 16) & 0xFF);
  sdo.body.data[7] = (UNS8)((abortCode >> 24) & 0xFF);
  ret = sendSDOMessage(d, sdo);

  return ret;
}

/*!
**
**
** @param d
** @param nodeId
** @param idx
** @param subIdx
** @param count
** @param dataType
** @param data
** @param Callback
** @param endianize
**
** @return
**/
INLINE UNS8 _writeNetworkDict (CO_Data* d, UNS8 nodeId, UNS16 idx,
		       UNS8 subIdx, UNS8 count, UNS8 dataType, void *data, SDOCallback_t Callback, void * Callbackpara, UNS8 endianize)
{
  UNS8 err;
  UNS8 SDOfound = 0;
  UNS8 line;
  s_transfer * lineptr;
  s_SDO sdo;    /* SDO to transmit */
  UNS8 i, j;
  UNS16     lastIndex;
  UNS16     offset;
  UNS8      *pNodeIdServer;
  UNS8      nodeIdServer;
  UNS32		cobidTx = 0;
  UNS32		cobidRx = 0;

  MSG_WAR(0x3AC0, "Send SDO to write in the dictionary of node : ", nodeId);
  MSG_WAR(0x3AC1, "                                   At idx : ", idx);
  MSG_WAR(0x3AC2, "                                   subIdx : ", subIdx);
  MSG_WAR(0x3AC3, "                                   nb bytes : ", count);

  /* Verify that there is no SDO communication yet. */
  err = getSDOlineOnUse(d, nodeId, SDO_CLIENT, &line, &lineptr);
  if (!err) {
    MSG_ERR(0x1AC4, "SDO error : Communication yet established. with node : ", nodeId);
    return 0xFF;
  }
  /* Taking the line ... */
  err = getSDOfreeLine( d, SDO_CLIENT, &line, &lineptr );
  if (err) {
    MSG_ERR(0x1AC5, "SDO error : No line free, too many SDO in progress. Aborted for node : ", nodeId);
    return (0xFF);
  }
  /* Check which SDO to use to communicate with the node */
  offset = d->firstIndex->SDO_CLT;
  lastIndex = d->lastIndex->SDO_CLT;
  if (offset == 0) {
    MSG_ERR(0x1AC6, "writeNetworkDict : No SDO client idx found", 0);
    return 0xFF;
  }
  i = 0;
  while (offset <= lastIndex) {
    if (d->objdict[offset].bSubCount <= 3) {
      MSG_ERR(0x1AC8, "Subindex 3  not found at idx ", 0x1280 + i);
      return 0xFF;
    }
    /* looking for the nodeId server */
    pNodeIdServer = (UNS8*) d->objdict[offset].pSubindex[3].pObject;
    nodeIdServer = *pNodeIdServer;
    MSG_WAR(0x1AD2, "idx : ", 0x1280 + i);
    MSG_WAR(0x1AD3, "nodeIdServer : ", nodeIdServer);

    if(nodeIdServer == nodeId) {
      SDOfound = 1;
	  cobidTx = *((UNS32 *)d->objdict[offset].pSubindex[1].pObject);
	  cobidRx = *((UNS32 *)d->objdict[offset].pSubindex[2].pObject);
	  break;
    }
    offset++;
    i++;
  } /* end while */
  if (!SDOfound) {
    MSG_ERR(0x1AC9, "SDO. Error. No client found to communicate with node : ", nodeId);
    return 0xFF;
  }
  MSG_WAR(0x3AD0,"        SDO client defined at idx  : ", 0x1280 + i);
#if USE_BLOCK_TRANSFER
  initSDOline(d, line, nodeId, idx, subIdx, count <= 21 ? SDO_DOWNLOAD_IN_PROGRESS: SDO_BLOCK_DOWNLOAD_IN_PROGRESS);
#else
  initSDOline(d, line, nodeId, idx, subIdx, SDO_DOWNLOAD_IN_PROGRESS);
#endif
  lineptr = &(d->transfers[line]);
  lineptr->count = count;
  lineptr->dataType = dataType;
  lineptr->cobidTx = cobidTx;
  lineptr->cobidRx = cobidRx;

  /* Copy data to transfers structure. */
  for (j = 0 ; j < count ; j++) {
# ifdef CANOPEN_BIG_ENDIAN
//wip: check for != visible_string instead of dataType == 0
    if (dataType != visible_string && endianize)
      lineptr->data[count - 1 - j] = ((char *)data)[j];
    else /* String of bytes. */
      lineptr->data[j] = ((char *)data)[j];
#  else
    lineptr->data[j] = ((char *)data)[j];
#  endif
  }
  /* Send the SDO to the server. Initiate download, cs=1. */
  sdo.cobid = cobidTx;
  if (count <= 4) { /* Expedited transfert */
    sdo.body.data[0] = (1 << 5) | ((4 - count) << 2) | 3;
    for (i = 4 ; i < 8 ; i++)
      sdo.body.data[i] = i < count+4 ? lineptr->data[i - 4] : 0;
    lineptr->offset = count;
  }
#if USE_BLOCK_TRANSFER  /* (blocktransfer does not work in scale)  */
  else if (count <= 21) { /** Normal transfert */
	sdo.body.data[0] = (INITIATE_DOWNLOAD_REQUEST << 5) | 1;
    sdo.body.data[4] = count; /* nb of byte to transmit. Max = 255. (canfestival2 limitation). */
    for (i = 5 ; i < 8 ; i++)
    	sdo.body.data[i] = 0;
  }
  else { /** Block transfer */
	  sdo.body.data[0] = (BLOCK_DOWNLOAD_REQUEST << 5) | 2;
	  sdo.body.data[4] = count; /* nb of byte to transmit. Max = 255. (canfestival2 limitation). */
	  for (i = 5 ; i < 8 ; i++)
		  sdo.body.data[i] = 0;

  }
#else
  else  { /** Normal transfert */
	sdo.body.data[0] = (INITIATE_DOWNLOAD_REQUEST << 5) | 1;
	sdo.body.data[4] = count; /* nb of byte to transmit. Max = 255. (canfestival2 limitation). */
	for (i = 5 ; i < 8 ; i++)
		sdo.body.data[i] = 0;
  }

#endif

  sdo.body.data[1] = idx & 0xFF;        /* LSB */
  sdo.body.data[2] = (idx >> 8) & 0xFF; /* MSB */
  sdo.body.data[3] = subIdx;

  lineptr->Callback = Callback;
  lineptr->Callbackpara = Callbackpara;

  err = sendSDOMessage(d, sdo);
  if (err) {
    MSG_ERR(0x1AD1, "SDO. Error while sending SDO to node : ", nodeId);
    /* release the line */
    resetSDOline(d, line);
    return 0xFF;
  }

  return 0;
}

/*!
**
**
** @param d
** @param nodeId
** @param idx
** @param subIdx
** @param count
** @param dataType
** @param data
**
** @return
**/
UNS8 writeNetworkDict (CO_Data* d, UNS8 nodeId, UNS16 idx,
		       UNS8 subIdx, UNS8 count, UNS8 dataType, void *data)
{
	return _writeNetworkDict (d, nodeId, idx, subIdx, count, dataType, data, NULL, NULL, 1);
}

/*!
**
**
** @param d
** @param nodeId
** @param idx
** @param subIdx
** @param count
** @param dataType
** @param data
** @param Callback
**
** @return
**/
UNS8 writeNetworkDictCallBack (CO_Data* d, UNS8 nodeId, UNS16 idx,
		       UNS8 subIdx, UNS8 count, UNS8 dataType, void *data, SDOCallback_t Callback, void * Callbackpara)
{
	return _writeNetworkDict (d, nodeId, idx, subIdx, count, dataType, data, Callback, Callbackpara, 1);
}

/*!
**
**
** @param d
** @param nodeId
** @param idx
** @param subIdx
** @param dataType
** @param Callback
**
** @return
**/
INLINE UNS8 _readNetworkDict (CO_Data* d, UNS8 nodeId, UNS16 idx, UNS8 subIdx, UNS8 dataType, SDOCallback_t Callback, void * Callbackpara)
{
  UNS8 err;
  UNS8 SDOfound = 0;
  UNS8 i;
  UNS8 line;
  s_transfer * lineptr;

  s_SDO sdo;    /* SDO to transmit */
  UNS8      *pNodeIdServer;
  UNS8      nodeIdServer;
  UNS16     offset;
  UNS16     lastIndex;
  UNS32		cobidTx = 0;
  UNS32		cobidRx = 0;

  MSG_WAR(0x3AD5, "Send SDO to read in the dictionary of node : ", nodeId);
  MSG_WAR(0x3AD6, "                                  At idx : ", idx);
  MSG_WAR(0x3AD7, "                                  subIdx : ", subIdx);


  /* Verify that there is no SDO communication yet. */
  err = getSDOlineOnUse(d, nodeId, SDO_CLIENT, &line, &lineptr);
  if (!err) {
    MSG_ERR(0x1AD8, "SDO error : Communication yet established. with node : ", nodeId);
    return 0xFF;
  }
  /* Taking the line ... */
  err = getSDOfreeLine( d, SDO_CLIENT, &line, &lineptr );
  if (err) {
    MSG_ERR(0x1AD9, "SDO error : No line free, too many SDO in progress. Aborted for node : ", nodeId);
    return (0xFF);
  }
  else
    MSG_WAR(0x3AE0, "Transmission on line : ", line);

  /* Check which SDO to use to communicate with the node */
  offset = d->firstIndex->SDO_CLT;
  lastIndex = d->lastIndex->SDO_CLT;
  if (offset == 0) {
    MSG_ERR(0x1AE1, "writeNetworkDict : No SDO client idx found", 0);
    return 0xFF;
  }
  i = 0;
  while (offset <= lastIndex) {
     if (d->objdict[offset].bSubCount <= 3) {
	 MSG_ERR(0x1AE2, "Subindex 3  not found at idx ", 0x1280 + i);
	 return 0xFF;
     }

     /* looking for the nodeId server */
     pNodeIdServer = (UNS8*) d->objdict[offset].pSubindex[3].pObject;
     nodeIdServer = *pNodeIdServer;

    MSG_WAR(0x3AD5, "Search Client entry : ", 0x1280 + i);

    MSG_WAR(0x3AD5, "Search Client nodeIdServer : ", nodeIdServer);
    if(nodeIdServer == nodeId) {
      SDOfound = 1;
	  cobidTx = *((UNS32 *)d->objdict[offset].pSubindex[1].pObject);
	  cobidRx = *((UNS32 *)d->objdict[offset].pSubindex[2].pObject);

      break;
    }
    offset++;
    i++;
  } /* end while */
  if (!SDOfound) {
    MSG_ERR(0x1AE3, "SDO. Error. No client found to communicate with node : ", nodeId);
    return 0xFF;
  }
  MSG_WAR(0x3AE4,"        SDO client defined at idx : ", 0x1280 + i);
  initSDOline(d, line, nodeId, idx, subIdx, SDO_UPLOAD_IN_PROGRESS);
  getSDOlineOnUse(d, nodeId, SDO_CLIENT, &line, &lineptr);
  lineptr = &(d->transfers[line]);
  lineptr->dataType = dataType;
  lineptr->cobidTx = cobidTx;
  lineptr->cobidRx = cobidRx;

  sdo.cobid = cobidTx;
  /* Send the SDO to the server. Initiate upload, cs=2. */
  sdo.body.data[0] = (INITIATE_UPLOAD_REQUEST << 5);
  sdo.body.data[1] = idx & 0xFF;        /* LSB */
  sdo.body.data[2] = (idx >> 8) & 0xFF; /* MSB */
  sdo.body.data[3] = subIdx;
  for (i = 4 ; i < 8 ; i++)
    sdo.body.data[i] = 0;
  lineptr->Callback = Callback;
  lineptr->Callbackpara = Callbackpara;
  err = sendSDOMessage(d, sdo);

  if (err) {
    MSG_ERR(0x1AE5, "SDO. Error while sending SDO to node : ", nodeId);
    /* release the line */
    resetSDOline(d, line);
    return 0xFF;
  }
  return 0;
}

/*!
**
**
** @param d
** @param nodeId
** @param idx
** @param subIdx
** @param dataType
**
** @return
**/
UNS8 readNetworkDict (CO_Data* d, UNS8 nodeId, UNS16 idx, UNS8 subIdx, UNS8 dataType)
{
	return _readNetworkDict (d, nodeId, idx, subIdx, dataType, NULL, NULL);
}

/*!
**
**
** @param d
** @param nodeId
** @param idx
** @param subIdx
** @param dataType
** @param Callback
**
** @return
**/
UNS8 readNetworkDictCallback (CO_Data* d, UNS8 nodeId, UNS16 idx, UNS8 subIdx, UNS8 dataType, SDOCallback_t Callback, void * Callbackpara)
{
	return _readNetworkDict (d, nodeId, idx, subIdx, dataType, Callback, Callbackpara);
}

/*!
**
**
** @param d
** @param nodeId
** @param data
** @param size
** @param abortCode
**
** @return
**/
UNS8 getReadResultNetworkDict (CO_Data* d, UNS8 nodeId, void* data, UNS8 *size,
			       UNS32 * abortCode)
{
  UNS8 i;
  UNS8 err;
  UNS8 line;
  s_transfer * lineptr;
  * size = 0;

  /* Looking for the line tranfert. */
  err = getSDOlineClosed(d, nodeId, SDO_CLIENT, &line, &lineptr);
  if (err) {
    MSG_ERR(0x1AF0, "SDO error : No line found for communication with node : ", nodeId);
    return SDO_ABORTED_INTERNAL;
  }
  if (lineptr->state != SDO_FINISHED)
    return lineptr->state;

  /* Transfert is finished. Put the value in the data. */
  * size = (UNS8)lineptr->count;
  for  ( i = 0 ; i < *size ; i++) {
# ifdef CANOPEN_BIG_ENDIAN
    if (lineptr->dataType != visible_string)
      ( (char *) data)[*size - 1 - i] = lineptr->data[i];
    else /* String of bytes. */
      ( (char *) data)[i] = lineptr->data[i];
# else
    ( (char *) data)[i] = lineptr->data[i];
# endif
  }
  return SDO_FINISHED;
}

/*!
**
**
** @param d
** @param nodeId
** @param abortCode
**
** @return
**/
UNS8 getWriteResultNetworkDict (CO_Data* d, UNS8 nodeId, UNS32 * abortCode)
{
  UNS8 line = 0;
  UNS8 err;
  s_transfer * lineptr;
  * abortCode = 0;
  /* Looking for the line tranfert. */
  err = getSDOlineOnUse(d, nodeId, SDO_CLIENT, &line, &lineptr);
  if (err) {
    MSG_ERR(0x1AF1, "SDO error : No line found for communication with node : ", nodeId);
    return SDO_ABORTED_INTERNAL;
  }
  * abortCode = lineptr->abortCode;
  return lineptr->state;
}


#if 0
UNS8 StartSDOBlockUploadCallback  (CO_Data* d, UNS8 nodeId, UNS16 idx,
                                   UNS8 subIdx, UNS32 count, UNS8 dataType, void *data, SDOCallback_t Callback)
{
  UNS8 err;
  //UNS8 SDOfound = 0;
  UNS8 i;
  UNS8 line;
  s_SDO sdo;    /* SDO to transmit */
  //UNS32      *pNodeIdServer;
  //UNS32      nodeIdServer;
  //UNS16     offset;
  //UNS16     lastIndex;
  UNS32 j;

  MSG_WAR(0x3AD5, "Send SDO to read in the dictionary of node : ", nodeId);
  MSG_WAR(0x3AD6, "                                  At idx : ", idx);
  MSG_WAR(0x3AD7, "                                  subIdx : ", subIdx);


  if (count > SDO_MAX_LENGTH_TRANSFERT)
  {
    // too long to transfer
    return 0xFF;
  }

  /* Verify that there is no SDO communication yet. */
  err = getSDOlineOnUse(d, nodeId, SDO_CLIENT, &line);
  if (!err)
  {
    MSG_ERR(0x1AD8, "SDO error : Communication yet established. with node : ", nodeId);
    return 0xFF;
  }
  /* Taking the line ... */
  err = getSDOfreeLine( d, SDO_CLIENT, &line );
  if (err)
  {
    MSG_ERR(0x1AD9, "SDO error : No line free, too many SDO in progress. Aborted for node : ", nodeId);
    return (0xFF);
  }
  else
    MSG_WAR(0x3AE0, "Transmission on line : ", line);

  MSG_WAR(0x3AE4,"        SDO client defined at idx  : ", 0x1280 + i);
  initSDOline(d, line, nodeId, idx, subIdx, SDO_BLOCK_TRANSFER_STARTING);
  getSDOlineOnUse(d, nodeId, SDO_CLIENT, &line);

  //lineptr->data = (char *)data;
  /* Copy data to transfers structure. */
  for (j = 0 ; j < count ; j++) {
# ifdef CANOPEN_BIG_ENDIAN
    if (dataType == 0 && endianize)
      lineptr->data[count - 1 - j] = ((char *)data)[j];
    else /* String of bytes. */
      lineptr->data[j] = ((char *)data)[j];
#  else
    lineptr->data[j] = ((char *)data)[j];
#  endif
  }

  lineptr->count = count;
  sdo.nodeId = nodeId;
  /* Send the SDO to the server. Initiate block upload, ccs=5. */
  lineptr->dataType = dataType;
  sdo.body.data[0] = (5 << 5);
  sdo.body.data[1] = idx & 0xFF;        /* LSB */
  sdo.body.data[2] = (idx >> 8) & 0xFF; /* MSB */
  sdo.body.data[3] = subIdx;
  sdo.body.data[4] = BLK_SIZE; /* blocksize, default 127*/
  for (i = 5 ; i < 8 ; i++)
  {
    sdo.body.data[i] = 0;
  }
  lineptr->Callback = Callback;
  err = sendSDO(d, SDO_CLIENT, sdo);
  if (err)
  {
    MSG_ERR(0x1AE5, "SDO. Error while sending SDO to node : ", nodeId);
    /* release the line */
    resetSDOline(d, line);
    return 0xFF;
  }
  return 0;
}


UNS8 StartSDOBlockDownloadCallback (CO_Data* d, UNS8 nodeId, UNS16 idx,
                                    UNS8 subIdx, UNS32 count, UNS8 dataType, void * data, SDOCallback_t Callback)
{
  UNS8 err;
  //UNS8 SDOfound = 0;
  UNS8 line;
  s_SDO sdo;    /* SDO to transmit */
  UNS8 i, j;
  UNS16     lastIndex;
  UNS16     offset;
  //  UNS32      *pNodeIdServer;
  //  UNS32      nodeIdServer;

  MSG_WAR(0x3AC0, "Send SDO to write in the dictionary of node : ", nodeId);
  MSG_WAR(0x3AC1, "                                   At idx : ", idx);
  MSG_WAR(0x3AC2, "                                   subIdx : ", subIdx);
  MSG_WAR(0x3AC3, "                                   nb bytes : ", count);

  if (count > SDO_MAX_LENGTH_TRANSFERT)
  {
    // too long to transfer
    return 0xFF;
  }

  /* Verify that there is no SDO communication yet. */
  err = getSDOlineOnUse(d, nodeId, SDO_CLIENT, &line);
  if (!err)
  {
    MSG_ERR(0x1AC4, "SDO error : Communication yet established. with node : ", nodeId);
    return 0xFF;
  }
  /* Taking the line ... */
  err = getSDOfreeLine( d, SDO_CLIENT, &line );
  if (err)
  {
    MSG_ERR(0x1AC5, "SDO error : No line free, too many SDO in progress. Aborted for node : ", nodeId);
    return (0xFF);
  }

  MSG_WAR(0x1AD3, "nodeIdServer : ", nodeId);

  MSG_WAR(0x3AD0,"        SDO client defined at idx  : ", 0x1280 + nodeId);
  initSDOline(d, line, nodeId, idx, subIdx, SDO_BLOCK_DOWNLOAD_IN_PROGRESS);
  lineptr->count = count;
  lineptr->dataType = dataType;

  /* Copy data to transfers structure. */
# ifdef CANOPEN_BIG_ENDIAN
  if (dataType == 0)
    lineptr->data[count - 1 - j] = ((char *)data)[j];
  else /* String of bytes. */
    lineptr->data[j] = ((char *)data)[j];
#  else
  lineptr->data[j] = ((char *)data)[j];
#  endif

  /* Send the SDO to the server. Initiate block download, ccs=6. */
  sdo.nodeId = nodeId;

  /* Normal transfert */
  sdo.body.data[0] = (6 << 5) | (1 << 1);
  sdo.body.data[1] = idx & 0xFF;        /* LSB */
  sdo.body.data[2] = (idx >> 8) & 0xFF; /* MSB */
  sdo.body.data[3] = subIdx;
  /* nb of bytes to transmit */
  sdo.body.data[4] = lineptr->count & 0xFF; /* LSB */
  sdo.body.data[5] = (lineptr->count >>  8) & 0xFF;
  sdo.body.data[6] = (lineptr->count >> 16) & 0xFF;
  sdo.body.data[7] = (lineptr->count >> 24) & 0xFF;  /* MSB */

  lineptr->Callback = Callback;

  err = sendSDO(d, SDO_CLIENT, sdo);
  if (err)
  {
    MSG_ERR(0x1AD1, "SDO. Error while sending SDO to node : ", nodeId);
    /* release the line */
    resetSDOline(d, line);
    return 0xFF;
  }


  return 0;
}
#endif

