/*
This file is part of CanFestival, a library implementing CanOpen Stack.

Copyright (C): Edouard TISSERANT and Francis DUPIN

See COPYING file for copyrights details.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/*!
 ** @file   sdoproc.c
 ** @author Wilhelm Pflueger
 ** @date
 **
 ** @brief	Process the SDO telegrams
 **
 **
 */


//#define DEBUG_WAR_CONSOLE_ON 1
#define DEBUG_ERR_CONSOLE_ON  1
//#define DEBUG_DBG_CONSOLE_ON 1
#define DEBUG_LAST_MESSAGE 1

#include <sys/time.h>
#include <time.h>

#include "objacces.h"
#include "sdo.h"
#include "canfestival.h"

/* Uncomment if your compiler does not support inline functions */
#define NO_INLINE

#ifdef NO_INLINE
#define INLINE
#else
#define INLINE inline
#endif

#define min(a,b) a<b?a:b


#define StopSDO_TIMER(id) \
MSG_WAR(0x3A05, "StopSDO_TIMER for line : ", id);\
d->transfers[id].timer = DelAlarm(d->transfers[id].timer);

#define StartSDO_TIMER(id) \
MSG_WAR(0x3A06, "StartSDO_TIMER for line : ", id);\
d->transfers[id].timer = SetAlarm(d,id,&SDOTimeoutAlarm,MS_TO_TIMEVAL(SDO_TIMEOUT_MS),0);

#define RestartSDO_TIMER(id) \
MSG_WAR(0x3A07, "restartSDO_TIMER for line : ", id);\
if(d->transfers[id].timer != TIMER_NONE) { StopSDO_TIMER(id) StartSDO_TIMER(id) }

#ifdef DEBUG_LAST_MESSAGE
Message LastTxMsg;
Message LastRxMsg;
struct timeval LastTxTime;
struct timeval LastRxTime;
#endif

#ifdef DEBUG_DBG_CONSOLE_ON
char dbgbuf[200];
static int dbgcount = 0;

char * dbghex (char *obuf, unsigned char* buf, unsigned char len)
{
	int i;

	sprintf (obuf, "%4.4d) ", dbgcount++);
	for (i=0; i<len; i++)
		sprintf (obuf+ 3*i + 6, "%02x ", buf[i]);

	return obuf;
}
#endif


/*!
 **
 **
 ** @param d
 ** @param whoami
 ** @param sdo
 **
 ** @return
 **/
UNS8 sendSDOMessage (CO_Data* d, s_SDO sdo)
{
	Message m;
	UNS8 i;
	MSG_WAR(0x3A38, "sendSDOMessage",0);
	if( !((d->nodeState == Operational) ||  (d->nodeState == Pre_operational )))
	{
		MSG_WAR(0x2A39, "unable to send the SDO (not in op or pre-op mode", d->nodeState);
		return 0xFF;
	}

	/* message copy for sending */
	m.cob_id.w = sdo.cobid;
	m.rtr = NOT_A_REQUEST;
	/* the length of SDO must be 8 */
	m.len = 8;
	for (i = 0 ; i < 8 ; i++)
	{
		m.data[i] =  sdo.body.data[i];
	}

#ifdef DEBUG_LAST_MESSAGE
	LastTxMsg = m;
	gettimeofday(&LastTxTime,0);
#endif
	return canSend(d->canHandle,&m);
}


UNS8 checkCobID (UNS8 * nodeId, UNS16 *nodeIdidx, CO_Data* d, Message *m)
{
	UNS8 whoami = SDO_UNKNOWN;  /* SDO_SERVER or SDO_CLIENT.*/

	UNS16 offset;
	UNS16 lastIndex;
	int   j;
	UNS32 CobId;

	UNS8 functionCode = (*m).cob_id.w >> 7;

	*nodeId = (*m).cob_id.w & 0x7f;

	if (  functionCode == SDOrx )
	{
		/* Server COBID */
		offset = d->firstIndex->SDO_SVR;
		lastIndex = d->lastIndex->SDO_SVR;
		j = 0;
		if(offset) while (offset <= lastIndex)
		{
			if (d->objdict[offset].bSubCount <= 1)
			{
				MSG_ERR(0x1A61, "Subindex 1  not found at idx ", 0x1200 + j);
				return SDO_UNKNOWN;
			}
			CobId = *( (UNS32*) d->objdict[offset].pSubindex[1].pObject);
			if ( CobId == (*m).cob_id.w )
			{
				whoami = SDO_SERVER;
				MSG_WAR(0x3A62, "checkCobID. I am server. idx : ", 0x1200 + j);
				/* In case of server, the node id of the client may be unknown. So we put the idx minus offset */
				/* 0x1200 where the cobid received is defined. */
				*nodeIdidx = j;
				MSG_WAR(0xfe, "nodeId: ", *nodeId);
				break;
			}
			j++;
			offset++;
		}

	}

	if ( functionCode == SDOtx )
	{
		/* Client COBID */
		offset = d->firstIndex->SDO_CLT;
		lastIndex = d->lastIndex->SDO_CLT;
		j = 0;
		if(offset) while (offset <= lastIndex)
		{
			if (d->objdict[offset].bSubCount <= 3)
			{
				MSG_ERR(0x1A63, "Subindex 3  not found at idx ", 0x1280 + j);
				return 0xFF;
			}
			/* a) Looking for the cobid received. */
			CobId = *((UNS32*) d->objdict[offset].pSubindex[2].pObject);
			if (CobId == (*m).cob_id.w )
			{
				/* b) cobid found, so reading the node id of the server. */
				*nodeId = *((UNS8*) d->objdict[offset].pSubindex[3].pObject);
				whoami = SDO_CLIENT;
				*nodeIdidx = j;
				MSG_WAR(0x3A64, "proceedSDO. I am client. idx : ", 0x1280 + j);
				MSG_WAR(0x3A65, "                 Server nodeId : ", *nodeId);
				break;
			}
			j++;
			offset++;
		}
	}


	/*--------------------------------------------------------------------------------------------
	Start checking of SDO telegram
	*/
	if (whoami == SDO_CLIENT)
	{
		MSG_WAR(0x3A68, "I am CLIENT. Received SDO from nodeId : ", *nodeId);
	}
	else if (whoami == SDO_CLIENT)
	{
		MSG_WAR(0x3A69, "I am SERVER. Received SDO cobId : ", (*m).cob_id.w);
	}

	return whoami;
}

UNS8 proceedSDOServerBlockDownload(CO_Data* d, Message *m, s_transfer *lineptr)
{
	UNS8  err;
	UNS32 nbBytes32;
	UNS8  nbBytes;
	int	  i;
	s_SDO sdo;    /* SDO to transmit */


	if ( lineptr->state == SDO_BLOCK_DOWNLOAD_IN_PROGRESS) /* next segment*/
	{
		/*sequence number is in  lineptr->toggle*/
		lineptr->toggle++;
		if( lineptr->toggle != (getSDO_block_seqno(m->data[0])) )
		{
			MSG_ERR(0x1A72, "SDO error : Sequence number error : ", getSDOt(m->data[0]));
			failedSDO(d, lineptr->nodeId, SDO_SERVER, 0, 0, SDOABT_INVALID_SEQ_NO);
			return SDO_ERROR;
		}
		MSG_WAR(0x3A69, "Block download in process, block: ", lineptr->toggle);
		nbBytes32 = lineptr->count - lineptr->offset;
		MSG_WAR(0x3A69, "nbBytes32: ", nbBytes32);
		nbBytes = min( (nbBytes32), 7);
		err = SDOtoLine(lineptr, nbBytes, (*m).data + 1);
		if (err)
		{
			failedSDO(d, lineptr->nodeId, SDO_SERVER, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
			return SDO_ERROR;
		}
		if( lineptr->toggle >= BLK_SIZE)
		{
			sdo.body.data[1] = BLK_SIZE; // ackseq
			lineptr->toggle = 0;
			if( getSDO_block_c((m->data[0])) )// last segment
			{
				lineptr->state = SDO_BLOCK_FINISHED;
			}
		}
		else
		{
			if( getSDO_block_c((m->data[0])) )// last segment
			{
				sdo.body.data[1] = lineptr->toggle;        /* ackseq */
				lineptr->state = SDO_BLOCK_FINISHED;
			}
			else // not last segment
			{
				return SDO_SUCCESS;
			}
		}
		sdo.body.data[2] = BLK_SIZE; /*set number of segments per block*/
		/*generate response*/
		/*Sending a SDO, cs=5*/
		sdo.cobid = lineptr->cobidTx; //*d->bDeviceNodeId; /* The node id of the server, (here it is the sender).*/
		sdo.body.data[0] = (BLOCK_DOWNLOAD_RESPONSE << 5) + 2;  /*ss = 2*/
		for (i = 3 ; i < 8 ; i++)
		{
			sdo.body.data[i] = 0;
		}
		sendSDOMessage(d, sdo);
		return SDO_SUCCESS;
	}
	else
		return SDO_UNHANDLED;
}

UNS8 proceedSDOClientBlockUpload(CO_Data* d, Message *m, s_transfer *lineptr)
{
	s_SDO sdo;    /* SDO to transmit */
	UNS32 nbBytes32;
	UNS8  nbBytes;
	UNS8  err;
	int   i;

	if( lineptr->state == SDO_BLOCK_UPLOAD_IN_PROGRESS)
	{
		/*sequence number is in  d->transfers[i].toggle*/
		lineptr->toggle++;
		if( lineptr->toggle != (getSDO_block_seqno(m->data[0])) )
		{
			MSG_ERR(0x1A72, "SDO error : Sequence number error : ", getSDOt(m->data[0]));
			failedSDO(d, lineptr->nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_INVALID_SEQ_NO);
			return SDO_ERROR;
		}
		nbBytes32 = lineptr->count - lineptr->offset;
		nbBytes = min( (nbBytes32), 7);
		err = SDOtoLine(lineptr, nbBytes, (*m).data + 1);
		if (err)
		{
			failedSDO(d, lineptr->nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
			return SDO_ERROR;
		}
		if( lineptr->toggle >= BLK_SIZE)
		{
			sdo.body.data[1] = BLK_SIZE; // ackseq
			lineptr->toggle = 0;
			if( getSDO_block_c((m->data[0])) )// last segment
			{
				lineptr->state = SDO_BLOCK_FINISHED;
			}
		}
		else
		{
			if( getSDO_block_c((m->data[0])) )// last segment
			{
				sdo.body.data[1] = lineptr->toggle;        /* ackseq */
				lineptr->state = SDO_BLOCK_FINISHED;
			}
			else // not last segment
			{
				return SDO_SUCCESS;
			}
		}
		sdo.body.data[2] = BLK_SIZE; /*set number of segments per block*/
		/*generate response*/
		/*Sending a SDO, cs=5*/
		sdo.cobid = lineptr->cobidTx; //*d->bDeviceNodeId; /* The node id of the server, (here it is the sender).*/
		sdo.body.data[0] = (BLOCK_UPLOAD_REQUEST << 5) + 2;  /*ss = 2*/
		for (i = 3 ; i < 8 ; i++)
		{
			sdo.body.data[i] = 0;
		}
		sendSDOMessage(d, sdo);
		return SDO_SUCCESS;

	}
	else
	{
		return SDO_UNHANDLED;
	}
}

UNS8 proceedSDOServerDownloadSegmentRequest(CO_Data* d, Message *m, UNS8 nodeId)
{
	UNS8  err;
	UNS8  lineno;
	UNS8  nbBytes;
	s_transfer *lineptr;
	s_SDO sdo;
	UNS32 errorCode; /* while reading or writing in the local object dictionary.*/
	int   i;

	/* Receiving a download segment data. */
	/* A SDO transfert should have been yet initiated. */
	err = getSDOlineOnUse( d, nodeId, SDO_SERVER, &lineno, &lineptr );
	if (!err)
		err = lineptr->state != SDO_DOWNLOAD_IN_PROGRESS;
	if (err) {
		MSG_ERR(0x1A70, "SDO error : Received download segment for unstarted trans. nodeId ", nodeId);
		failedSDO(d, nodeId, SDO_SERVER, 0, 0, SDOABT_LOCAL_CTRL_ERROR);
		return 0xFF;
	}
	/* Reset the wathdog */
	RestartSDO_TIMER(lineno)
	MSG_WAR(0x3A71, "Received SDO download segment defined at nodeId ", nodeId);

	/* Toggle test. */
	if (lineptr->toggle != getSDOt(m->data[0])) {
		MSG_ERR(0x1A72, "SDO error : Toggle error : ", getSDOt(m->data[0]));
		failedSDO(d, nodeId, SDO_SERVER, lineptr->idx, lineptr->subIdx, SDOABT_TOGGLE_NOT_ALTERNED);
		return 0xFF;
	}
	/* Nb of data to be downloaded */
	nbBytes = 7 - getSDOn3(m->data[0]);
	/* Store the data in the transfert structure. */
	err = SDOtoLine(lineptr, nbBytes, (*m).data + 1);
	if (err) {
		failedSDO(d, nodeId, SDO_SERVER, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
		return 0xFF;
	}
	/* Sending the SDO response, CS = 1 */
	sdo.cobid = lineptr->cobidTx; //*d->bDeviceNodeId; /* The node id of the server, (here it is the sender). */
	sdo.body.data[0] = (DOWNLOAD_SEGMENT_RESPONSE << 5) | (lineptr->toggle << 4);
	for (i = 1 ; i < 8 ; i++)
		sdo.body.data[i] = 0;
	MSG_WAR(0x3A73, "SDO. Send response to download request defined at nodeId", nodeId);
	sendSDOMessage(d, sdo);
	/* Inverting the toggle for the next segment. */
	lineptr->toggle = ! lineptr->toggle & 1;
	/* If it was the last segment, */
	if (getSDOc(m->data[0])) {
		/* Transfering line data to object dictionary. */
		/* The code does not use the "d" of initiate frame. So it is safe if e=s=0 */
		errorCode = SDOlineToObjdict(d, lineno);
		if (errorCode) {
			MSG_ERR(0x1A54, "SDO error : Unable to copy the data in the object dictionary", 0);
			failedSDO(d, nodeId, SDO_SERVER, lineptr->idx, lineptr->subIdx, errorCode);
			return 0xFF;
		}
		/* Release of the line */
		terminateSDOline(d, lineno);
		MSG_WAR(0x3A74, "SDO. End of download at nodeId", nodeId);
	}

	return err;
}

UNS8 proceedSDOClientUploadSegmentResponse(CO_Data* d, Message *m, UNS8 nodeId)
{
	UNS8  err;
	UNS8  lineno;
	UNS8  nbBytes;
	s_transfer *lineptr;
	s_SDO sdo;
	int i;

	/* I am CLIENT */
	/* It is a request for a previous upload segment. We should find a line opened for this.*/
	err = getSDOlineOnUse( d, nodeId, SDO_CLIENT, &lineno, &lineptr);

	if (!err)
			err = lineptr->state != SDO_UPLOAD_IN_PROGRESS;
	if (err) {
		MSG_ERR(0x1A75, "SDO error : Received segment response for unknown trans. from nodeId", nodeId);
		failedSDO(d, nodeId, SDO_CLIENT, 0, 0, SDOABT_LOCAL_CTRL_ERROR);
		return 0xFF;
	}
	/* Reset the wathdog */
	RestartSDO_TIMER(lineno)
	/* test of the toggle; */
	if (lineptr->toggle != getSDOt(m->data[0])) {
		MSG_ERR(0x1A76, "SDO error : Received segment response Toggle error. from nodeId", nodeId);
		failedSDO(d, nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_TOGGLE_NOT_ALTERNED);
		return 0xFF;
	}
	/* nb of data to be uploaded */
	nbBytes = 7 - getSDOn3(m->data[0]);
	/* Storing the data in the line structure. */
	err = SDOtoLine(lineptr, nbBytes, (*m).data + 1);
	if (err) {
		failedSDO(d, nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
		return 0xFF;
	}
	/* Inverting the toggle for the next segment. */
	lineptr->toggle = ! lineptr->toggle & 1;
	/* If it was the last segment,*/
	if ( getSDOc(m->data[0])) {
		/* Put in state finished */
		/* The code is safe for the case e=s=0 in initiate frame. */
		StopSDO_TIMER(lineno)
		lineptr->state = SDO_FINISHED;
		// if(lineptr->Callback) (*lineptr->Callback)(d,nodeId);
		terminateSDOline(d, lineno);

		MSG_WAR(0x3A77, "SDO. End of upload from node : ", nodeId);
	}
	else { /* more segments to receive */
		/* Sending the request for the next segment. */
		sdo.cobid = lineptr->cobidTx;
		sdo.body.data[0] = (3 << 5) | (lineptr->toggle << 4);
		for (i = 1 ; i < 8 ; i++)
			sdo.body.data[i] = 0;
		sendSDOMessage(d, sdo);
		MSG_WAR(0x3A78, "SDO send upload segment request to nodeId", nodeId);
	}

	return err;
}

UNS8 proceedSDOServerDownloadInitRequest(CO_Data* d, Message *m, UNS8 nodeId, UNS8 nodeIdidx)
{
	UNS8  err;
	UNS8  lineno;
	UNS8  nbBytes;
	s_transfer *lineptr;
	s_SDO sdo;
	int i;
	UNS16 idx;
	UNS8  subIdx;
	UNS16 offset;
	UNS32 errorCode; /* while reading or writing in the local object dictionary.*/

	/* I am SERVER */
	/* Receive of an initiate download */
	idx = getSDOindex(m->data[1],m->data[2]);
	subIdx = getSDOsubIndex(m->data[3]);
	MSG_WAR(0x3A79, "Received SDO Initiate Download (to store data) defined at nodeId", nodeId);
	MSG_WAR(0x3A80, "Writing at idx : ", idx);
	MSG_WAR(0x3A80, "Writing at subIdx : ", subIdx);

	/* Search if a SDO transfert have been yet initiated */
	err = getSDOlineOnUse( d, nodeId, SDO_SERVER, &lineno, &lineptr );
	if (! err) {
		MSG_ERR(nodeId, "SDO error : Transmission yet started:", subIdx);
		failedSDO(d, nodeId, SDO_SERVER, idx, subIdx, SDOABT_LOCAL_CTRL_ERROR);
		return 0xFF;
	}

	/* No line on use. Great ! */
	/* Try to open a new line. */
	err = getSDOfreeLine( d, SDO_SERVER, &lineno, &lineptr );

	if (err) {
		MSG_ERR(0x1A82, "SDO error : No line free, too many SDO in progress. Aborted.", nodeId);
		failedSDO(d, nodeId, SDO_SERVER, idx, subIdx, SDOABT_LOCAL_CTRL_ERROR);
		return 0xFF;
	}

	initSDOline(d, lineno, nodeId, idx, subIdx, SDO_DOWNLOAD_IN_PROGRESS);
	offset = d->firstIndex->SDO_SVR;

	lineptr->cobidTx = *((UNS32 *)d->objdict[offset+nodeIdidx].pSubindex[2].pObject);
	lineptr->cobidRx = *((UNS32 *)d->objdict[offset+nodeIdidx].pSubindex[1].pObject);
	lineptr->nodeIdidx = nodeIdidx;

	/* Check if OD is writeable */
	if (getSDOe(m->data[0])) { /* If SDO expedited */
		/* nb of data to be downloaded */
		nbBytes = 4 - getSDOn2(m->data[0]);
		if (getSDOn2x(m->data[0]) > 3) {
			nbBytes = 0;
			MSG_ERR(0x1A72, "SDO error : wrong command spec : %", m->data[0]);
			MSG_DBG(nodeId, dbghex(dbgbuf, m->data, 8), 0);
		}
	} else {
		nbBytes = m->data[4]; /* Transfert limited to 255 bytes. */
	}

	if (nbBytes > 0)
	{
		errorCode =  checkODentry (d, idx, subIdx, (UNS32) nbBytes, 1 /* check access */);

		if (errorCode != OD_SUCCESSFUL)
		{
			MSG_ERR(0x1A83, "SDO error : OD entry blocked.", errorCode);
			MSG_ERR(idx, "<-- idx  - subindex -->", subIdx);

			failedSDO(d, nodeId, SDO_SERVER, idx, subIdx, errorCode);
			return 0xFF;
		}
	}

	/* Prepare the response telegram */
	sdo.cobid = lineptr->cobidTx; //* The node id of the server, (here it is the sender).*/
	sdo.body.data[0] = INITIATE_DOWNLOAD_RESPONSE << 5;
	sdo.body.data[1] = idx & 0xFF;        /* LSB */
	sdo.body.data[2] = (idx >> 8) & 0xFF; /* MSB */
	sdo.body.data[3] = subIdx;
	for (i = 4 ; i < 8 ; i++)
		sdo.body.data[i] = 0;

	if (getSDOe(m->data[0])) { /* If SDO expedited */
		/* nb of data to be downloaded */
		// already known: nbBytes = 4 - getSDOn2(m->data[0]);
		/* Storing the data in the line structure. */

		if (nbBytes > 0)
		{
			lineptr->count = nbBytes;
			err = SDOtoLine(lineptr, nbBytes, (*m).data + 4);

			if (err) {
				failedSDO(d, nodeId, SDO_SERVER, idx, subIdx, SDOABT_GENERAL_ERROR);
				return 0xFF;
			}
		}
		// send response telegram
		sendSDOMessage(d, sdo);

		/* SDO expedited -> transfert finished. Data can be stored in the dictionary. */
		/*The line will be reseted when it is downloading in the dictionary. */
		MSG_WAR(0x3A83, "SDO Initiate Download is an expedited transfert. Finished.: ", nodeId);

		if (nbBytes > 0)
		{
			/* Transfering line data to object dictionary. */
			errorCode = SDOlineToObjdict(d, lineno);
			if (errorCode) {
				MSG_ERR(0x1A84, "SDO error : Unable to copy the data in the object dictionary", 0);
				failedSDO(d, nodeId, SDO_SERVER, idx, subIdx, errorCode);
				return 0xFF;
			}
		}
		/* Release of the line. */
		terminateSDOline(d, lineno);
	}
	else {/* So, if it is not an expedited transfert */
		if (getSDOs(m->data[0])) {
			/* TODO : if e and s = 0, not reading m->data[4] but put nbBytes = 0 */
			nbBytes = m->data[4]; /* Transfert limited to 255 bytes. */
			err = setSDOlineRestBytes(lineptr, nbBytes);
			if (err) {
				failedSDO(d, nodeId, SDO_SERVER, idx, subIdx, SDOABT_GENERAL_ERROR);
				return 0xFF;
			}
			// send response telegram
			sendSDOMessage(d, sdo);
		}
	}

	return err;
}

UNS8 proceedSDOClientDownloadSegmentResponse(CO_Data* d, Message *m, UNS8 nodeId)
{
	UNS8  err;
	UNS8  lineno;
	UNS8  nbBytes;
	s_transfer *lineptr;
	s_SDO sdo;
	int i;

	/* I am CLIENT */
	/* It is a response for a previous download segment. We should find a line opened for this. */
	err = getSDOlineOnUse( d, nodeId, SDO_CLIENT, &lineno, &lineptr);

	if (!err)
		err = (lineptr->state != SDO_DOWNLOAD_IN_PROGRESS);
	if (err) {
		MSG_ERR(0x1A85, "SDO error : Received segment response for unknown trans. from nodeId", nodeId);
		failedSDO(d, nodeId, SDO_CLIENT, 0, 0, SDOABT_LOCAL_CTRL_ERROR);
		return 0xFF;
	}
	/* Reset the wathdog */
	RestartSDO_TIMER(lineno)
	/* test of the toggle; */
	if (lineptr->toggle != getSDOt(m->data[0])) {
		MSG_ERR(0x1A86, "SDO error : Received segment response Toggle error. from nodeId", nodeId);
		failedSDO(d, nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_TOGGLE_NOT_ALTERNED);
		return 0xFF;
	}

	/* End transmission or downloading next segment. We need to know if it will be the last one. */
	getSDOlineRestBytes(d, lineno, &nbBytes);
	if (nbBytes == 0) {
		MSG_WAR(0x3A87, "SDO End download. segment response received. OK. from nodeId", nodeId);
		StopSDO_TIMER(lineno)
		lineptr->state = SDO_FINISHED;
		// if(lineptr->Callback) (*lineptr->Callback)(d,nodeId);
		terminateSDOline(d, lineno);
		return 0x00;
	}
	/* At least one transfer to send.	*/
	if (nbBytes > 7) {
		/* several segments to download.*/
		/* code to send the next segment. (cs = 0; c = 0) */
		lineptr->toggle = ! lineptr->toggle & 1;
		sdo.cobid = lineptr->cobidTx; /* The server node Id; */
		sdo.body.data[0] = (lineptr->toggle << 4);
		err = lineToSDO(d, lineno, 7, sdo.body.data + 1);
		if (err) {
			failedSDO(d, nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
			return 0xFF;
		}
	}
	else {
		/* Last segment. */
		/* code to send the last segment. (cs = 0; c = 1)*/
		lineptr->toggle = ! lineptr->toggle & 1;
		sdo.cobid = lineptr->cobidTx; /* The server node Id; */
		sdo.body.data[0] = (lineptr->toggle << 4) | ((7 - nbBytes) << 1) | 1;
		err = lineToSDO(d, lineno, nbBytes, sdo.body.data + 1);
		if (err) {
			failedSDO(d, nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
			return 0xFF;
		}
		for (i = nbBytes + 1 ; i < 8 ; i++)
			sdo.body.data[i] = 0;
	}
	MSG_WAR(0x3A88, "SDO sending download segment to nodeId", nodeId);
	sendSDOMessage(d, sdo);
	return err;
}

UNS8 proceedSDOServerUploadInitRequest(CO_Data* d, Message *m, UNS8 nodeId, UNS8 nodeIdidx)
{
	UNS8  err;
	UNS8  lineno;
	UNS8  nbBytes;
	s_transfer *lineptr;
	s_SDO sdo;
	int i;
	UNS16 idx;
	UNS8  subIdx;
	UNS16 offset;
	UNS32 errorCode; /* while reading or writing in the local object dictionary.*/

	/* I am SERVER */
	/* Receive of an initiate upload.*/
	idx = getSDOindex(m->data[1],m->data[2]);
	subIdx = getSDOsubIndex(m->data[3]);
	MSG_WAR(0x3A89, "Received SDO Initiate upload (to send data) defined at nodeId:", nodeId);
	MSG_WAR(0x3A90, "Reading at idx : ", idx);
	MSG_WAR(0x3A91, "Reading at subIdx : ", subIdx);
	/* Search if a SDO transfert have been yet initiated*/
	err = getSDOlineOnUse( d, nodeId, SDO_SERVER, &lineno, &lineptr );
	if (! err) {
		MSG_ERR(0x1A92, "SDO error : Transmission yet started at line : ", lineno);
		MSG_WAR(0x3A93, "nodeId = ", nodeId);
		failedSDO(d, nodeId, SDO_SERVER, idx, subIdx, SDOABT_LOCAL_CTRL_ERROR);
		return 0xFF;
	}
	/* No line on use. Great !*/
	/* Try to open a new line.*/
	err = getSDOfreeLine( d, SDO_SERVER, &lineno, &lineptr );

	if (err) {
		MSG_ERR(0x1A71, "SDO error : No line free, too many SDO in progress. Aborted.", 0);
		failedSDO(d, nodeId, SDO_SERVER, idx, subIdx, SDOABT_LOCAL_CTRL_ERROR);
		return 0xFF;
	}
	initSDOline(d, lineno, nodeId, idx, subIdx, SDO_UPLOAD_IN_PROGRESS);
	offset = d->firstIndex->SDO_SVR;

	lineptr->cobidTx = *((UNS32 *)d->objdict[offset+nodeIdidx].pSubindex[2].pObject);
	lineptr->cobidRx = *((UNS32 *)d->objdict[offset+nodeIdidx].pSubindex[1].pObject);
	lineptr->nodeIdidx = nodeIdidx;


	/* Transfer data from dictionary to the line structure. */
	errorCode = objdictToSDOline(d, lineno);

	if (errorCode) {
		MSG_ERR(0x1A94, "SDO error : Unable to copy the data from object dictionary. Err code : ", errorCode);
		failedSDO(d, nodeId, SDO_SERVER, idx, subIdx, errorCode);
		return 0xFF;
	}
	/* Preparing the response.*/
	getSDOlineRestBytes(d, lineno, &nbBytes);	/* Nb bytes to transfer ? */
	sdo.cobid = lineptr->cobidTx; /* The server node Id; */
	if (nbBytes > 4) {
		/* normal transfert. (segmented). */
		/* code to send the initiate upload response. (cs = 2) */
		sdo.body.data[0] = (2 << 5) | 1;
		sdo.body.data[1] = idx & 0xFF;        /* LSB */
		sdo.body.data[2] = (idx >> 8) & 0xFF; /* MSB */
		sdo.body.data[3] = subIdx;
		sdo.body.data[4] = nbBytes; /* Limitation of canfestival2 : Max tranfert is 256 bytes.*/
		/* It takes too much memory to upgrate to 2^32 because the size of data is also coded */
		/* in the object dictionary, at every idx and subindex. */
		for (i = 5 ; i < 8 ; i++)
			sdo.body.data[i] = 0;
		MSG_WAR(0x3A95, "SDO. Sending normal upload initiate response defined at nodeId:", nodeId);
		sendSDOMessage(d, sdo);
	}
	else {
		/* Expedited upload. (cs = 2 ; e = 1) */
		sdo.body.data[0] = (2 << 5) | ((4 - nbBytes) << 2) | 3;
		sdo.body.data[1] = idx & 0xFF;        /* LSB */
		sdo.body.data[2] = (idx >> 8) & 0xFF; /* MSB */
		sdo.body.data[3] = subIdx;
		err = lineToSDO(d, lineno, nbBytes, sdo.body.data + 4);
		if (err) {
			failedSDO(d, nodeId, SDO_SERVER, idx, subIdx, SDOABT_GENERAL_ERROR);
			return 0xFF;
		}
		for (i = 4 + nbBytes ; i < 8 ; i++)
			sdo.body.data[i] = 0;
		MSG_WAR(0x3A96, "SDO. Sending expedited upload initiate response defined at nodeId:", nodeId);
		sendSDOMessage(d, sdo);
		/* Release the line.*/
		terminateSDOline(d, lineno);
	}

	return err;
}

UNS8 proceedSDOClientUploadInitResponse(CO_Data* d, Message *m, UNS8 nodeId)
{
	UNS8  err;
	UNS8  lineno;
	UNS8  nbBytes;
	s_transfer *lineptr;
	s_SDO sdo;
	int i;

	/* I am CLIENT */
	/* It is the response for the previous initiate upload request.*/
	/* We should find a line opened for this. */
	err = getSDOlineOnUse( d, nodeId, SDO_CLIENT, &lineno, &lineptr);

	if (!err)
		err = lineptr->state != SDO_UPLOAD_IN_PROGRESS;
	if (err) {
		MSG_ERR(0x1A97, "SDO error : Received response for unknown upload request from nodeId", nodeId);
		failedSDO(d, nodeId, SDO_CLIENT, 0, 0, SDOABT_LOCAL_CTRL_ERROR);
		return 0xFF;
	}
	/* Reset the wathdog */
	RestartSDO_TIMER(lineno)

	if (getSDOe(m->data[0])) { /* If SDO expedited */
		/* nb of data to be uploaded */
		nbBytes = 4 - getSDOn2(m->data[0]);
		/* Storing the data in the line structure. */
		err = SDOtoLine(lineptr, nbBytes, (*m).data + 4);
		if (err) {
			failedSDO(d, nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
			return 0xFF;
		}
		/* SDO expedited -> transfert finished. data are available via  getReadResultNetworkDict(). */
		MSG_WAR(0x3A98, "SDO expedited upload finished. Response received from node : ", nodeId);
		StopSDO_TIMER(lineno)
		lineptr->count = nbBytes;
		lineptr->state = SDO_FINISHED;
		// if(lineptr->Callback) (*lineptr->Callback)(d,nodeId);
		terminateSDOline(d, lineno);
		return 0;
	}
	else { /* So, if it is not an expedited transfert */
		/* Storing the nb of data to receive. */
		if (getSDOs(m->data[0])) {
			nbBytes = m->data[4]; /* Remember the limitation to 255 bytes to transfert */
			err = setSDOlineRestBytes(lineptr, nbBytes);
			if (err) {
				failedSDO(d, nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
				return 0xFF;
			}
		}
		/* Requesting next segment. (cs = 3) */
		sdo.cobid = lineptr->cobidTx;
		sdo.body.data[0] = UPLOAD_SEGMENT_REQUEST << 5;
		for (i = 1 ; i < 8 ; i++)
			sdo.body.data[i] = 0;
		MSG_WAR(0x3A99, "SDO. Sending upload segment request to node : ", nodeId);
		sendSDOMessage(d, sdo);
	}


	return err;
}

UNS8 proceedSDOServerUploadSegmentRequest(CO_Data* d, Message *m, UNS8 nodeId)
{
	UNS8  err;
	UNS8  lineno;
	UNS8  nbBytes;
	s_transfer *lineptr;
	s_SDO sdo;
	int   i;

	/* Receiving a upload segment. */
	/* A SDO transfert should have been yet initiated. */
	err = getSDOlineOnUse( d, nodeId, SDO_SERVER, &lineno, &lineptr );
	if (!err)
		err = lineptr->state != SDO_UPLOAD_IN_PROGRESS;
	if (err) {
		MSG_ERR(0x1AA0, "SDO error : Received upload segment for unstarted trans. idx 0x1200 + ", nodeId);
		failedSDO(d, nodeId, SDO_SERVER, 0, 0, SDOABT_LOCAL_CTRL_ERROR);
		return 0xFF;
	}
	/* Reset the wathdog */
	RestartSDO_TIMER(lineno)
			MSG_WAR(0x3AA1, "Received SDO upload segment defined at idx 0x1200 + ", nodeId);
	/* Toggle test.*/
	if (lineptr->toggle != getSDOt(m->data[0])) {
		MSG_ERR(0x1AA2, "SDO error : Toggle error : ", getSDOt(m->data[0]));
		failedSDO(d, nodeId, SDO_SERVER, lineptr->idx, lineptr->subIdx, SDOABT_TOGGLE_NOT_ALTERNED);
		return 0xFF;
	}
	/* Uploading next segment. We need to know if it will be the last one. */
	getSDOlineRestBytes(d, lineno, &nbBytes);
	if (nbBytes > 7) {
		/* The segment to transfer is not the last one.*/
		/* code to send the next segment. (cs = 0; c = 0) */
		sdo.cobid = lineptr->cobidTx; /* The server node Id; */
		sdo.body.data[0] = (lineptr->toggle << 4);
		err = lineToSDO(d, lineno, 7, sdo.body.data + 1);
		if (err) {
			failedSDO(d, nodeId, SDO_SERVER, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
			return 0xFF;
		}
		/* Inverting the toggle for the next tranfert. */
		lineptr->toggle = ! lineptr->toggle & 1;
		MSG_WAR(0x3AA3, "SDO. Sending upload segment defined at idx 0x1200 + ", nodeId);
		sendSDOMessage(d, sdo);
	}
	else {
		/* Last segment. */
		/* code to send the last segment. (cs = 0; c = 1) */
		sdo.cobid = lineptr->cobidTx; /** The server node Id; */
		sdo.body.data[0] = (lineptr->toggle << 4) | ((7 - nbBytes) << 1) | 1;
		err = lineToSDO(d, lineno, nbBytes, sdo.body.data + 1);
		if (err) {
			failedSDO(d, nodeId, SDO_SERVER, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
			return 0xFF;
		}
		for (i = nbBytes + 1 ; i < 8 ; i++)
			sdo.body.data[i] = 0;
		MSG_WAR(0x3AA4, "SDO. Sending last upload segment defined at idx 0x1200 + ", nodeId);
		sendSDOMessage(d, sdo);
		/* Release the line */
		terminateSDOline(d, lineno);
	}

	return err;
}

UNS8 proceedSDOClientDownloadInitResponse(CO_Data* d, Message *m, UNS8 nodeId)
{
	UNS8  err;
	UNS8  lineno;
	UNS8  nbBytes;
	s_transfer *lineptr;
	s_SDO sdo;
	int i;

	/* I am CLIENT */
	/* It is the response for the previous initiate download request. */
	/* We should find a line opened for this. */
	err = getSDOlineOnUse( d, nodeId, SDO_CLIENT, &lineno, &lineptr);

	if (!err)
		err = lineptr->state != SDO_DOWNLOAD_IN_PROGRESS;
	if (err) {
		failedSDO(d, nodeId, SDO_CLIENT, 0, 0, SDOABT_LOCAL_CTRL_ERROR);
		return 0xFF;
	}
	/* Reset the watchdog */
	RestartSDO_TIMER(lineno)
	/* End transmission or requesting  next segment. */
	getSDOlineRestBytes(d, lineno, &nbBytes);
	if (nbBytes == 0) {
		MSG_WAR(0x3AA6, "SDO End download expedited. Response received. from nodeId", nodeId);
		StopSDO_TIMER(lineno)
		lineptr->state = SDO_FINISHED;
		//if(lineptr->Callback) (*lineptr->Callback)(d,nodeId);
		terminateSDOline(d, lineno);
		return 0x00;
	}
	if (nbBytes > 7) {
		/* more than one request to send */
		/* code to send the next segment. (cs = 0; c = 0)	*/
		sdo.cobid = lineptr->cobidTx; /** The server node Id; */
		sdo.body.data[0] = (lineptr->toggle << 4);
		err = lineToSDO(d, lineno, 7, sdo.body.data + 1);
		if (err) {
			failedSDO(d, nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
			return 0xFF;
		}
	}
	else {
		/* Last segment.*/
		/* code to send the last segment. (cs = 0; c = 1)	*/
		sdo.cobid = lineptr->cobidTx; /* The server node Id; */
		sdo.body.data[0] = (lineptr->toggle << 4) | ((7 - nbBytes) << 1) | 1;
		err = lineToSDO(d, lineno, nbBytes, sdo.body.data + 1);
		if (err) {
			failedSDO(d, nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
			return 0xFF;
		}
		for (i = nbBytes + 1 ; i < 8 ; i++)
			sdo.body.data[i] = 0;
	}
	MSG_WAR(0x3AA7, "SDO sending download segment to nodeId", nodeId);
	sendSDOMessage(d, sdo);


	return err;
}

UNS8 proceedSDOServerAbortRequest(CO_Data* d, Message *m, UNS8 nodeId)
{
	UNS8  err;
	UNS8  lineno;
	UNS32 abortCode;
	s_transfer * lineptr;

	abortCode = (*m).data[4] |
			((UNS32)m->data[5] << 8) |
			((UNS32)m->data[6] << 16) |
			((UNS32)m->data[7] << 24);
	/* Received SDO abort. */
	/* Looking for the line concerned. */

	err = getSDOlineOnUse( d, nodeId, SDO_SERVER, &lineno, &lineptr );
	if (!err) {
		lineptr->abortCode = abortCode;
		terminateSDOline( d, lineno );
		MSG_DBG(0x3AA8, "SD0. Received SDO abort. Line released. Code : ", abortCode);
	}
	else
		MSG_DBG(0x3AA9, "SD0. Received SDO abort. No line found. Code : ", abortCode);
	/* Tips : The end user has no way to know that the server node has received an abort SDO. */
	/* Its is ok, I think.*/

	return err;
}

UNS8 proceedSDOClientAbortRequest(CO_Data* d, Message *m, UNS8 nodeId)
{
	UNS8  err;
	UNS8  lineno;
	s_transfer *lineptr;
	UNS32 abortCode;

	abortCode = (*m).data[4] |
			((UNS32)m->data[5] << 8) |
			((UNS32)m->data[6] << 16) |
			((UNS32)m->data[7] << 24);
	/* Received SDO abort. */
	/* Looking for the line concerned. */
	err = getSDOlineOnUse( d, nodeId, SDO_CLIENT, &lineno, &lineptr );

	if (!err) {
		/* The line *must* be released by the core program. */
		StopSDO_TIMER(lineno)
				lineptr->state = SDO_ABORTED_RCV;
		lineptr->abortCode = abortCode;
		MSG_DBG(0x3AB0, "SD0. Received SDO abort. Line state ABORTED. Code : ", abortCode);
#if 0
		if(lineptr->Callback)
			(*lineptr->Callback)(d,nodeId);
		else
			resetSDOline( d, lineno );
#endif
		terminateSDOline(d, lineno);
	}
	else
		MSG_WAR(0x3AB1, "SD0. Received SDO abort. No line found. Code : ", abortCode);

	return err;
}

UNS8 proceedSDOServerBlockUploadRequest(CO_Data* d, Message *m, UNS8 nodeId, UNS8 nodeIdidx)
{
	UNS8  err;
	UNS8  lineno;
	UNS8  nbBytes;
	UNS32 nbBytes32;
	UNS16 offset;
	UNS16 idx;
	UNS8  subIdx;
	s_transfer *lineptr;
	s_SDO sdo;
	int   i;

	if( (m->data[0] & 1))
	{
		err = getSDOlineOnUse( d, nodeId, SDO_SERVER, &lineno, &lineptr);
		if ( err )
		{
			MSG_ERR(0x1AA5, "SDO error : Block Upload: no line found for this nodeId:", nodeId);
			failedSDO(d, nodeId, SDO_SERVER, 0, 0, SDOABT_LOCAL_CTRL_ERROR);
			return 0xFF;
		}
		if( lineptr->state == SDO_BLOCK_FINISHED )/* end block upload request */
		{
			resetSDOline(d, lineno);
			return 0;
		}
	}

	switch(m->data[0] & 3) /*get client subcommand p. 55 in CANopen Spec.*/
	{
		/*----------------------------------------------------------------
		initiate upload request
		*/
		case 0:
		{
			/* number of segments per block(1-127), not used yet, 127 is default value*/
			/*  blksize = m->data[4]*/
			/* pst = m->data[5] */
			idx = getSDOindex(m->data[1],m->data[2]);
			subIdx = getSDOsubIndex(m->data[3]);
			MSG_WAR(0x3A89, "Received SDO Initiate block upload (to send data) defined at idx 0x1200 + ",
					nodeId);
			MSG_WAR(0x3A90, "Reading at idx : ", idx);
			MSG_WAR(0x3A91, "Reading at subIdx : ", subIdx);
			/* Search if a SDO transfert have been yet initiated*/
			err = getSDOlineOnUse( d, nodeId, SDO_SERVER, &lineno, &lineptr );
			if (! err)
			{
				MSG_ERR(0x1A92, "SDO error : Transmission yet started at line : ", lineno);
				MSG_WAR(0x3A93, "nodeId = ", nodeId);
				failedSDO(d, nodeId, SDO_SERVER, idx, subIdx, SDOABT_LOCAL_CTRL_ERROR);
				return 0xFF;
			}
			/* No line on use. Great !*/
			/* Try to open a new line.*/
			err = getSDOfreeLine( d, SDO_SERVER, &lineno,&lineptr );
			if (err)
			{
				MSG_ERR(0x1A71, "SDO error : No line free, too many SDO in progress. Aborted.", 0);
				failedSDO(d, nodeId, SDO_SERVER, idx, subIdx, SDOABT_LOCAL_CTRL_ERROR);
				return 0xFF;
			}
			initSDOline(d, lineno, nodeId, idx, subIdx, SDO_BLOCK_UPLOAD_IN_PROGRESS);
			offset = d->firstIndex->SDO_SVR;

			lineptr->cobidTx = *((UNS32 *)d->objdict[offset+nodeIdidx].pSubindex[2].pObject);
			lineptr->cobidRx = *((UNS32 *)d->objdict[offset+nodeIdidx].pSubindex[1].pObject);
			lineptr->nodeIdidx = nodeIdidx;



			/* Preparing the response.*/
			//      getSDOlineRestBytes(d, line, &nbBytes); /* Nb bytes to transfer ? */
			sdo.cobid = lineptr->cobidTx; /* The server node Id; */
			/* normal transfert. (segmented). */
			/* code to send the initiate upload response. (cs = 6) */
			sdo.body.data[0] = (6 << 5) | 2; /* data size is indicated*/
			sdo.body.data[1] = idx & 0xFF;        /* LSB */
			sdo.body.data[2] = (idx >> 8) & 0xFF; /* MSB */
			sdo.body.data[3] = subIdx;
			/* size */
			sdo.body.data[4] = lineptr->count & 0xFF; /* LSB */
			sdo.body.data[5] = (lineptr->count >>  8) & 0xFF;
			sdo.body.data[6] = (lineptr->count >> 16) & 0xFF;
			sdo.body.data[7] = (lineptr->count >> 24) & 0xFF;  /* MSB */

			MSG_WAR(0x3A95, "SDO. Sending block upload initiate response defined at idx 0x1200 + ", nodeId);
			sendSDOMessage(d, sdo);
			break;
		}
			 /*----------------------------------------------------------------
			  block upload response
			 */
		case 2:
		{
			err = getSDOlineOnUse( d, nodeId, SDO_SERVER, &lineno, &lineptr );
			if ( err )
			{
				MSG_ERR(0x1AA5, "SDO error : Block upload response nof line in use for nodeId: ", nodeId);
				failedSDO(d, nodeId, SDO_SERVER, 0, 0, SDOABT_LOCAL_CTRL_ERROR);
				return 0xFF;
			}

			if(lineptr->state != SDO_BLOCK_UPLOAD_IN_PROGRESS)
			{
				return 0XFF;
			}
			if(lineptr->toggle == m->data[1])
			{/* block was received successfully */
				if( lineptr->count == lineptr->offset)
				{ /* all segments sent */
					/* Preparing the response.*/
					sdo.cobid = lineptr->cobidTx; /* The server node Id; */
					nbBytes = 7 - (lineptr->count % 7); /* number of bytes in
					the last segment of the last block that do not contain data.*/
					sdo.body.data[0] = (6 << 5) | (nbBytes << 2) | 1;
					for (i = 1 ; i < 8 ; i++)
					{
						sdo.body.data[i] = 0;
					}
					sendSDOMessage(d, sdo);
					lineptr->state = SDO_BLOCK_FINISHED;
		// closeSDOtransfer (d, nodeId, SDO_CLIENT);
					return 0;
				}
				else
				{/* send next block*/
					lineptr->toggle = 0;
					sdo.cobid = lineptr->cobidTx; /* The server node Id; */
					nbBytes32 = lineptr->count - lineptr->offset;
					nbBytes = min( (nbBytes32), 7);
					while (nbBytes32)
					{
						err = lineToSDO(d, lineno, nbBytes, sdo.body.data + 1);
						if (err)
						{
							failedSDO(d, nodeId, SDO_SERVER, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
							return 0xFF;
						}
						lineptr->toggle++; //increment sequence number
						if(nbBytes32 <= 7)
						{
			// last segment
							sdo.body.data[0] = lineptr->toggle + (1 << 7);
						}
						else
						{
							sdo.body.data[0] = lineptr->toggle;
						}
						sendSDOMessage(d, sdo);
						nbBytes32 -= nbBytes;
						nbBytes = min( (nbBytes32), 7);
						if(lineptr->toggle >= BLK_SIZE)
						{
			//            lineptr->toggle = 0;
							return 0;
						}
					} // end while
				}
			}
			else
			{/* retransmit segments with number > m->data[1]*/
				/* calculate offset */
				nbBytes = lineptr->count % 7;
				if (nbBytes && (lineptr->count == lineptr->offset))
				{
					lineptr->offset -= (7*(lineptr->toggle
							- m->data[1]) + nbBytes - 7 );
				}
				else
				{
					lineptr->offset -= 7*(lineptr->toggle - m->data[1]);
				}
				lineptr->toggle = m->data[1];
				sdo.cobid = lineptr->cobidTx; /* The server node Id; */
				nbBytes32 = lineptr->count - lineptr->offset;
				nbBytes = min( (nbBytes32), 7);
				while (nbBytes32)
				{
					err = lineToSDO(d, lineno, nbBytes, sdo.body.data + 1);
					if (err)
					{
						failedSDO(d, nodeId, SDO_SERVER, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
						return 0xFF;
					}
					lineptr->toggle++; //increment sequence number
					if(nbBytes32 <= 7)
					{
			// last segment
						sdo.body.data[0] = lineptr->toggle + (1 << 7);
					}
					else
					{
						sdo.body.data[0] = lineptr->toggle;
					}
					sendSDOMessage(d, sdo);
					nbBytes32 -= nbBytes;
					nbBytes = min( (nbBytes32), 7);
					if(lineptr->toggle >= BLK_SIZE)
					{
			//            lineptr->toggle = 0;
						return 0;
					}
				}
			}
			break;
		}
		case 3:      /* start upload request*/
		{
			err = getSDOlineOnUse( d, nodeId, SDO_SERVER, &lineno, &lineptr );
			if (err)
			{
				MSG_ERR(0x1AA5, "SDO error : Start upload REQUEST no line in use for nodeId: ", nodeId);
				failedSDO(d, nodeId, SDO_SERVER, 0, 0, SDOABT_LOCAL_CTRL_ERROR);
				return 0xFF;
			}

			if(lineptr->state != SDO_BLOCK_UPLOAD_IN_PROGRESS)
			{
				return 0XFF;
			}
			/* transfer 1. block to client*/
			sdo.cobid = lineptr->cobidTx; /* The server node Id; */
			nbBytes32 = lineptr->count;
			nbBytes = min( (nbBytes32), 7);
			while (nbBytes32)
			{
				err = lineToSDO(d, lineno, nbBytes, sdo.body.data + 1);
				if (err)
				{
					failedSDO(d, nodeId, SDO_SERVER, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
					return 0xFF;
				}
				lineptr->toggle++; //increment sequence number
				if(nbBytes32 <= 7)
				{
	// last segment
					sdo.body.data[0] = lineptr->toggle + (1 << 7);
				}
				else
				{
					sdo.body.data[0] = lineptr->toggle;
				}
				sendSDOMessage(d, sdo);
				nbBytes32 -= nbBytes;
				nbBytes = min( (nbBytes32), 7);
				if(lineptr->toggle >= BLK_SIZE)
				{
				//            lineptr->toggle = 0;
					return 0;
				}
			} // end while
			//       lineptr->state = SDO_BLOCK_FINISHED;
			return 0;
			break;
		}

		default:
			err = 0;
	}

	return err;
}

UNS8 proceedSDOClientBlockDownloadResponse(CO_Data* d, Message *m, UNS8 nodeId)
{
	UNS8  err;
	UNS8  lineno;
	UNS8  nbBytes;
	UNS32 nbBytes32;
	s_transfer *lineptr;
	s_SDO sdo;
	int i;

	err = getSDOlineOnUse( d, nodeId, SDO_CLIENT, &lineno, &lineptr );

	if ( err )
	{
		MSG_ERR(0x1AA5, "SDO error : Block Download response: no line in use for nodeId: ", nodeId);
		failedSDO(d, nodeId, SDO_CLIENT, 0, 0, SDOABT_LOCAL_CTRL_ERROR);
		return 0xFF;
	}

	switch(m->data[0] & 3) /*get server subcommand p. 51 in CANopen Spec.*/
	{
		case 0: /*it is response to init block download*/
			if(lineptr->state != SDO_BLOCK_DOWNLOAD_IN_PROGRESS)
			{
				return 0XFF;
			}
			/* transfer 1. block to server*/
			sdo.cobid = lineptr->cobidTx;
			nbBytes32 = lineptr->count;
			nbBytes = min( (nbBytes32), 7);
			while (nbBytes32)
			{
				MSG_DBG(nbBytes32,"Block transfer to another server", nodeId);
				err = lineToSDO(d, lineno, nbBytes, sdo.body.data + 1);
				if (err)
				{
					failedSDO(d, nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
					return 0xFF;
				}
				lineptr->toggle++; //increment sequence number
				if(nbBytes32 <= 7)
				{
            // last segment
					sdo.body.data[0] = lineptr->toggle + (1 << 7);
				}
				else
				{
					sdo.body.data[0] = lineptr->toggle;
				}
				sendSDOMessage(d, sdo);
				nbBytes32 -= nbBytes;
				nbBytes = min( (nbBytes32), 7);
				if(lineptr->toggle >= BLK_SIZE)
				{
            //            lineptr->toggle = 0;
					return 0;
				}
			} // end while
			return 0;
			break;


		case 2:               /* block download response */
			if(lineptr->state != SDO_BLOCK_DOWNLOAD_IN_PROGRESS)
			{
				return 0XFF;
			}
			if(lineptr->toggle == m->data[1])
			{/* block was received successfully */
				if( lineptr->count == lineptr->offset)
				{ /* all segments sent */
					/* Preparing the response.*/
					sdo.cobid = lineptr->cobidTx;
					nbBytes = 7 - (lineptr->count % 7); /* number of bytes in
					the last segment of the last block that do not contain data.*/
					sdo.body.data[0] = (6 << 5) | (nbBytes << 2) | 1;
					for (i = 1 ; i < 8 ; i++)
					{
						sdo.body.data[i] = 0;
					}
					sendSDOMessage(d, sdo);
					lineptr->state = SDO_BLOCK_FINISHED;
		// closeSDOtransfer (d, nodeId, SDO_CLIENT);
					return 0;
				}
				else
				{/* send next block*/
					lineptr->toggle = 0;
					sdo.cobid = lineptr->cobidTx; /* The server node Id; */
					nbBytes32 = lineptr->count - lineptr->offset;
					nbBytes = min( (nbBytes32), 7);
					while (nbBytes32)
					{
						err = lineToSDO(d, lineno, nbBytes, sdo.body.data + 1);
						if (err)
						{
							failedSDO(d, nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
							return 0xFF;
						}
						lineptr->toggle++; //increment sequence number
						if(nbBytes32 <= 7)
						{
			// last segment
							sdo.body.data[0] = lineptr->toggle + (1 << 7);
						}
						else
						{
							sdo.body.data[0] = lineptr->toggle;
						}
						sendSDOMessage(d, sdo);
						nbBytes32 -= nbBytes;
						nbBytes = min( (nbBytes32), 7);
						if(lineptr->toggle >= BLK_SIZE)
						{
			//            lineptr->toggle = 0;
							return 0;
						}
					} // end while
				}
			}
			else
			{/* retransmitt segments with number > m->data[1]*/
				/* calculate offset */
				nbBytes = lineptr->count % 7;
				if (nbBytes && (lineptr->count == lineptr->offset))
				{
					lineptr->offset -= (7*(lineptr->toggle
							- m->data[1]) + nbBytes - 7 );
				}
				else
				{
					lineptr->offset -= 7*(lineptr->toggle - m->data[1]);
				}
				lineptr->toggle = m->data[1];
				sdo.cobid = lineptr->cobidTx;
				nbBytes32 = lineptr->count - lineptr->offset;
				nbBytes = min( (nbBytes32), 7);
				while (nbBytes32)
				{
					err = lineToSDO(d, lineno, nbBytes, sdo.body.data + 1);
					if (err)
					{
						failedSDO(d, nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
						return 0xFF;
					}
					lineptr->toggle++; //increment sequence number
					if(nbBytes32 <= 7)
					{
			// last segment
						sdo.body.data[0] = lineptr->toggle + (1 << 7);
					}
					else
					{
						sdo.body.data[0] = lineptr->toggle;
					}
					sendSDOMessage(d, sdo);
					nbBytes32 -= nbBytes;
					nbBytes = min( (nbBytes32), 7);
					if(lineptr->toggle >= BLK_SIZE)
					{
			//            lineptr->toggle = 0;
						return 0;
					}
				}
			}
			break;

		case 1:               /* end block download  */
			if(lineptr->state != SDO_BLOCK_FINISHED)
			{
				return 0XFF;
			}
			// TODO: assume that the resetSDO will be invoked later .....
			if(lineptr->Callback)
			{
				(*lineptr->Callback)(d,lineno, nodeId,lineptr);
			}


	} /* end switch */

	return err;
}

UNS8 proceedSDOServerBlockDownloadRequest(CO_Data* d, Message *m, UNS8 nodeId, UNS8 nodeIdidx)
{
	UNS8  err;
	UNS8  lineno;
	UNS32 nbBytes32;
	UNS16 offset;
	UNS16 idx;
	UNS8  subIdx;
	UNS32 errorCode; /* while reading or writing in the local object dictionary.*/


	s_transfer *lineptr;
	s_SDO sdo;
	int   i;

	if(getSDO_block_cs(m->data[0]))
	{
		/* Receive of an end block download request*/
		/*generate response*/
		/*Sending a SDO, cs=5*/
		err = getSDOlineOnUse( d, nodeId, SDO_SERVER, &lineno, &lineptr );

		if (!err)
			err = lineptr->state != SDO_BLOCK_FINISHED;
		if (err) {
			MSG_ERR(0x1A73, "SDO error : Block transfer not correct finished, NodeId ", nodeId);
			failedSDO (d, nodeId, SDO_SERVER, lineptr->idx, lineptr->subIdx, SDOABT_LOCAL_CTRL_ERROR);
			return 0xFF;
		}

		sdo.cobid = lineptr->cobidTx; //*d->bDeviceNodeId; /* The node id of the server, (here it is the sender).*/
		sdo.body.data[0] = (BLOCK_DOWNLOAD_RESPONSE << 5)+1;
		for (i = 1 ; i < 8 ; i++)
		{
			sdo.body.data[i] = 0;
		}
		sendSDOMessage(d, sdo);
		// xBPIStr_To_RS232();
		/* Reset the watchdog */
		RestartSDO_TIMER(lineno)
		MSG_WAR(0x3A71, "Received SDO download segment defined at idx 0x1200 + ", nodeId);
		errorCode = SDOlineToObjdict(d, lineno);
		if (errorCode) {
			MSG_ERR(0x1A84, "SDO error : Unable to copy the data in the object dictionary", 0);
			failedSDO(d, nodeId, SDO_SERVER, lineptr->idx, lineptr->subIdx, errorCode);
			return 0xFF;
		}
		/* Release of the line */
		resetSDOline(d, lineno);
	}
	else
	{
		/* Receive of an initiate block download */
		idx = getSDOindex(m->data[1],m->data[2]);
		subIdx = getSDOsubIndex(m->data[3]);
		MSG_WAR(0x3A79, "Received SDO Initiate Block Download (to store data) defined at idx 0x1200 + ", nodeId);
		MSG_WAR(0x3A80, "Writing at idx : ", idx);
		MSG_WAR(0x3A80, "Writing at subIdx : ", subIdx);

		/* Search if a SDO transfert have been yet initiated */
		err = getSDOlineOnUse( d, nodeId, SDO_SERVER, &lineno, &lineptr );
		if (! err)
		{
			MSG_ERR(0x1A81, "SDO error : Transmission yet started.", 0);
			failedSDO(d, nodeId, SDO_SERVER, idx, subIdx, SDOABT_LOCAL_CTRL_ERROR);
			return 0xFF;
		}

		/* No line on use. Great ! */
		/* Try to open a new line. */
		err = getSDOfreeLine( d, SDO_SERVER, &lineno, &lineptr );
		if (err)
		{
			MSG_ERR(0x1A82, "SDO error : No line free, too many SDO in progress. Aborted.", 0);
			failedSDO(d, nodeId, SDO_SERVER, idx, subIdx, SDOABT_LOCAL_CTRL_ERROR);
			return 0xFF;
		}
		initSDOline(d, lineno, nodeId, idx, subIdx, SDO_BLOCK_DOWNLOAD_IN_PROGRESS);
		offset = d->firstIndex->SDO_SVR;

		lineptr->cobidTx = *((UNS32 *)d->objdict[offset+nodeIdidx].pSubindex[2].pObject);
		lineptr->cobidRx = *((UNS32 *)d->objdict[offset+nodeIdidx].pSubindex[1].pObject);
		lineptr->nodeIdidx = nodeIdidx;

		/* Get the data length */
		if (getSDO_block_s(m->data[0]))
		{
			/* TODO : if s = 0, not reading m->data[4] but put nbBytes = 0 */
			nbBytes32 = getSDO_block_len(m->data[4],m->data[5],m->data[6],m->data[7]);
			MSG_WAR(0x3A80, "nbBytes32: ", nbBytes32);
			MSG_WAR(0x3A80, "count: ", lineptr->count);
			lineptr->count = nbBytes32;
		}
		else
			nbBytes32 = 0;

		/* Check if OD is writeable */
		errorCode =  checkODentry (d, idx, subIdx, nbBytes32, 1 /* check access */);

		if (errorCode != OD_SUCCESSFUL)
		{
			MSG_ERR(nbBytes32, "SDO error : OD entry write not allowed.", errorCode);
			failedSDO(d, lineptr->nodeId, SDO_SERVER, idx, subIdx, errorCode);
			return 0xFF;
		}

		/*generate response*/
		/*Sending a SDO, cs=5*/
		sdo.cobid = lineptr->cobidTx; //*d->bDeviceNodeId; /* The node id of the server, (here it is the sender).*/
		sdo.body.data[0] = BLOCK_DOWNLOAD_RESPONSE << 5;
		sdo.body.data[1] = idx & 0xFF;        /* LSB */
		sdo.body.data[2] = (idx >> 8) & 0xFF; /* MSB */
		sdo.body.data[3] = subIdx;
		sdo.body.data[4] = BLK_SIZE;       /*set number of segments per block*/

		for (i = 5 ; i < 8 ; i++)
		{
			sdo.body.data[i] = 0;
		}
		sendSDOMessage(d, sdo);
	}

	return err;
}

UNS8 proceedSDOClientBlockUploadResponse(CO_Data* d, Message *m, UNS8 nodeId)
{
	UNS8  err;
	UNS8  lineno;
	UNS32 nbBytes32;
	s_transfer *lineptr;
	s_SDO sdo;
	int i;

	/* We should find a line opened for this. */
	err = getSDOlineOnUse( d, nodeId, SDO_CLIENT, &lineno, &lineptr);
	if (err)
	{
		MSG_ERR(0x1AA5, "SDO error : Block Upload Response: no line in use for nodeId: ", nodeId);
		failedSDO(d, nodeId, SDO_CLIENT, 0, 0, SDOABT_LOCAL_CTRL_ERROR);
		return 0xFF;
	}
	if( lineptr->state == SDO_BLOCK_TRANSFER_STARTING)
	{/* It is the response for the previous initiate block upload request.*/
		if (!m->data[0]& 1)
		{/* ss == 0*/
			MSG_ERR(0x1A97, "SDO error : Received response for unknown block upload request from nodeId", nodeId);
			failedSDO(d, nodeId, SDO_CLIENT, 0, 0, SDOABT_LOCAL_CTRL_ERROR);
			return 0xFF;
		}
		/* Storing the nb of data to receive. */
		if (getSDO_block_s(m->data[0]))/* data set size is indicated*/
		{
			/* TODO : if s = 0, not reading m->data[4] but put nbBytes = 0 */
			nbBytes32 = getSDO_block_len(m->data[4],m->data[5],m->data[6],m->data[7]);
			if( lineptr->count < nbBytes32)
			{
				failedSDO(d, nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
				return 0xff;
			}
			else
			{
				lineptr->count = nbBytes32;
			}

		}
		lineptr->state = SDO_BLOCK_UPLOAD_IN_PROGRESS;
		/*generate response*/
		/*Sending a SDO, cs=5*/
		sdo.cobid = lineptr->cobidTx;
		sdo.body.data[0] = (BLOCK_UPLOAD_REQUEST << 5) | 3;
		for (i = 1 ; i < 8 ; i++)
		{
			sdo.body.data[i] = 0;
		}
		sendSDOMessage(d, sdo);
		return 0;
	}
	/*ss=1, end block upload */
	if(( (m->data[0]& 3) != 1) && (lineptr->state != SDO_BLOCK_FINISHED))
	{
		failedSDO(d, nodeId, SDO_CLIENT, lineptr->idx, lineptr->subIdx, SDOABT_GENERAL_ERROR);
		return 0xFF;
	}
	/*generate response*/
	/*Sending a SDO, cs=5*/
	sdo.cobid = lineptr->cobidTx;
	sdo.body.data[0] = (BLOCK_UPLOAD_REQUEST << 5) | 1;
	for (i = 1 ; i < 8 ; i++)
	{
		sdo.body.data[i] = 0;
	}
	sendSDOMessage(d, sdo);

	// TODO: assume that resetSDO will be called later....
	if(lineptr->Callback)
	{
		(*lineptr->Callback)(d,lineno,nodeId,lineptr);
	}


	return err;
}

/*!
**
**
** @param d
** @param m
**
** @return
** */

UNS8 proceedSDO (CO_Data* d, Message *m)
{
	UNS8 err;
	UNS8 nodeId = 0;  /* The node from which the SDO is received */
	UNS16 nodeIdidx = 0;
	UNS8 whoami;  /* SDO_SERVER or SDO_CLIENT.*/
	UNS8 i;
#ifdef DEBUG_LAST_MESSAGE
	LastRxMsg = *m;
    gettimeofday(&LastRxTime,0);
#endif
	/*-------------------------------------------------------------------------------------------
	Check Telegram
	*/
	MSG_WAR(0x3A60, "proceedSDO ", 0);

	whoami = checkCobID (&nodeId, &nodeIdidx, d, m);

	if (whoami == SDO_UNKNOWN)
		return 0xFF;

	/* Test if the size of the SDO is ok */
	if ( (*m).len != 8)
	{
		MSG_ERR(0x1A67, "Error size SDO. CobId  : ", (*m).cob_id.w);
		failedSDO(d, nodeId, whoami, 0, 0, SDOABT_GENERAL_ERROR);
		return 0xFF;
	}


	/*-------------------------------------------------------------------------------------
	 Testing if SDO block transfer in progress
	*/
	err = SDO_UNHANDLED;
	for(i=0;i< SDO_MAX_SIMULTANEOUS_TRANSFERTS;i++)
	{
		if (d->transfers[i].state != SDO_RESET
				  && d->transfers[i].nodeId == nodeId)
		{
			if (whoami == SDO_SERVER)
				err = proceedSDOServerBlockDownload(d, m, &(d->transfers[i]));

			if (whoami == SDO_CLIENT)
				err = proceedSDOClientBlockUpload(d, m, &(d->transfers[i]));

			// >>>> stop loop -> we already found one line.
			break;
		}

	}

	// terminate routine if message is already handled.
	if (err != SDO_UNHANDLED)
		return err;

  /*--------------------------------------------------------------------------------------------------
  Testing other command specifier
  */
  /* Testing the command specifier */
  /* Allowed : cs = 0, 1, 2, 3, 4. (=  all except those for block tranfer). */
  /* cs = other : Not allowed -> abort. */

	if (whoami == SDO_SERVER)
	{
		/* I am a server */
		switch (getSDOcs(m->data[0])) {
			case DOWNLOAD_SEGMENT_REQUEST:
				err = proceedSDOServerDownloadSegmentRequest(d, m, nodeId);
				break;

			case INITIATE_DOWNLOAD_REQUEST:
				err = proceedSDOServerDownloadInitRequest(d, m, nodeId, nodeIdidx);
				break;

			case INITIATE_UPLOAD_REQUEST:
				err = proceedSDOServerUploadInitRequest(d, m, nodeId, nodeIdidx);
				break;

			case UPLOAD_SEGMENT_REQUEST:
				err = proceedSDOServerUploadSegmentRequest(d, m, nodeId);
				break;

			case ABORT_TRANSFER_REQUEST:
				err = proceedSDOServerAbortRequest(d, m, nodeId);
				break;

			case BLOCK_UPLOAD_REQUEST:
				err = proceedSDOServerBlockUploadRequest(d, m, nodeId, nodeIdidx);
				break;

			case BLOCK_DOWNLOAD_REQUEST:
				err = proceedSDOServerBlockDownloadRequest(d, m, nodeId, nodeIdidx);
				break;

			default:
				/* Error : Unknown cs */
				MSG_ERR(0x1AB2, "SDO. Received unknown command specifier : ", getSDOcs(m->data[0]));
				err = 0xFF;

		}
	}
	else
	{  /* I am a client */

		switch (getSDOcs(m->data[0])) {
			case UPLOAD_SEGMENT_RESPONSE:
				err = proceedSDOClientUploadSegmentResponse(d, m, nodeId);
				break;

			case DOWNLOAD_SEGMENT_RESPONSE:
				err = proceedSDOClientDownloadSegmentResponse(d, m, nodeId);
				break;

			case INITIATE_UPLOAD_RESPONSE:
				err = proceedSDOClientUploadInitResponse(d, m, nodeId);
				break;

			case INITIATE_DOWNLOAD_RESPONSE:
				err = proceedSDOClientDownloadInitResponse(d, m, nodeId);
				break;

			case ABORT_TRANSFER_REQUEST:
				err = proceedSDOClientAbortRequest(d, m, nodeId);
				break;

			case BLOCK_DOWNLOAD_RESPONSE:
				err = proceedSDOClientBlockDownloadResponse(d, m, nodeId);
				break;

			case BLOCK_UPLOAD_RESPONSE:
				err = proceedSDOClientBlockUploadResponse(d, m, nodeId);
				break;

			default:
				/* Error : Unknown cs */
				MSG_ERR(0x1AB2, "SDO. Received unknown command specifier : ", getSDOcs(m->data[0]));
				err = 0xFF;


		}
	}

	return err;
}

