/*!
 ** @file   timerqueue.c
 ** @author Wilhelm Pflueger
 ** @date   Tue Jun  5 09:32:32 2007
 **
 ** @brief  Timer implementation using timer delta queues
 **
 **
 */

//#define DEBUG_WAR_CONSOLE_ON 
#define DEBUG_ERR_CONSOLE_ON 
//#define DEBUG_DBG_CONSOLE_ON

#include <applicfg.h>
#include "timer.h"

typedef struct struct_timerqueue
{
	struct struct_timerqueue *prev;
	struct struct_timerqueue *next;
	TIMEVAL 	expireTime;
	s_timer_entry	timer;
} t_timerqueue;


t_timerqueue tq[MAX_NB_TIMER] = {{NULL, NULL, 0, {TIMER_FREE, NULL, NULL, 0, 0, 0},}};


void timer_enable(void);
void timer_disable(void);
void stop_oneshot(void);
void start_oneshot(void);
void queue_delete(t_timerqueue *tq_old);
void queue_insert(t_timerqueue *tq_new);
void start_timer(int number,unsigned int timevalue,void (*routine)(void));
void stop_timer(int number);
void timerqueue_enable(void);

void TimeDispatch(void)
{
	t_timerqueue *tqnext;
	TimerCallback_t timer_routine = NULL;
	TIMEVAL currentTime;
	CO_Data* d;
	UNS32 id;
	UNS8 timer_state;

	// Should all have been done by EnterMutex())
	// EnterTimerMutex();
	tqnext = tq->next;
// 	if (tqnext != NULL)
// 	{
// 		tqnext->expireTime = 0;
// 	}

	setTimer(TIMEVAL_MAX);
	
// 	if (tqnext == NULL)
// 	{
// 		setTimer(TIMEVAL_MAX);
// 	}

	while (tqnext != NULL)
	{
		/* get timevalue since signal occured */
		currentTime = getCurrentTime();
		timer_routine = NULL;
		/* Timer already expired? */
		if (tqnext->expireTime - currentTime <= 0)
		{
			/* remove entry from head of queue */
			tq->next = tqnext->next;
			if (tq->next != NULL)
				tq->next->prev = tq;
			
			
			timer_routine = tqnext->timer.callback;
			d = tqnext->timer.d;
			id = tqnext->timer.id;
			timer_state = tqnext->timer.state;
			
			tqnext->prev = NULL;
			
			/* retrigger periodic timer*/
			if (tqnext->timer.interval)
			{
				tqnext->expireTime = currentTime + tqnext->timer.interval;
				tqnext->timer.state = TIMER_ARMED;
				queue_insert(tqnext);
			}
			else
			{
				tqnext->timer.state = TIMER_FREE;
			}
			
			/* execute timerroutine */
			if (timer_state == TIMER_ARMED
						 && timer_routine)
				timer_routine(d, id);
			
			/* get next queue entry */
			tqnext = tq->next;
			
			
		}
		else
		{

			setTimer(tqnext->expireTime - currentTime);
			/* stop searching for expired timers */
			tqnext = NULL;
		}
		
	}
	
	// LeaveTimerMutex();
	
}

/*!                                                                                                
 ** -------  Use this to declare a new alarm ------                                                                                                
 **                                                                                                 
 ** @param d                                                                                        
 ** @param id                                                                                       
 ** @param callback                                                                                 
 ** @param value                                                                                    
 ** @param period                                                                                   
 **                                                                                                 
 ** @return                                                                                         
 **/   
TIMER_HANDLE SetAlarm(CO_Data* d, UNS32 id, TimerCallback_t callback, TIMEVAL value, TIMEVAL period)
{
	TIMER_HANDLE i;
	TIMER_HANDLE handle = TIMER_NONE;
	TIMEVAL currentTime;

	currentTime = getCurrentTime();
	// EnterTimerMutex();
	/* in order to decide new timer setting we have to run over all timer rows */
	for(i=1; i < MAX_NB_TIMER; i++)
	{
		if (tq[i].prev == NULL)
		{
			t_timerqueue *tqnew = tq + i;

			handle = i;
			if (callback) /* something to store */
			{	/* just store */
				tqnew->timer.callback = callback;
				tqnew->timer.d = d;
				tqnew->timer.id = id;
				tqnew->timer.val = value;
				tqnew->timer.interval = period;
				tqnew->timer.state = TIMER_ARMED;
				tqnew->expireTime = currentTime + value;
				queue_insert(tqnew);
				
				// Start timer if the timer is the next one.
				if (tqnew->prev == &tq[0])
					setTimer(value);
			}
			
			break;
		}
		
	}
	//  LeaveTimerMutex();

	return handle;
}


/*!                                                                                                
 **  -----  Use this to remove an alarm ----                                                                                             
 **                                                                                                 
 ** @param handle                                                                                   
 **                                                                                                 
 ** @return                                                                                         
 **/  
TIMER_HANDLE DelAlarm(TIMER_HANDLE handle)
{
	t_timerqueue *tqhandle;
	
	MSG_WAR(0x3320, "DelAlarm. handle = ", handle);
	if(handle != TIMER_NONE && handle < MAX_NB_TIMER)
	{
		// EnterTimerMutex();
		
		tqhandle = &tq[handle];
		tqhandle->timer.state = TIMER_FREE;
		// if this is the head of the queue: correct the timer
		if (tqhandle->prev == &tq[0])
		{
			// if the only timer in queue --> defer it as long as possible
			if (tqhandle->next == NULL)
				setTimer(TIMEVAL_MAX);
			else
			{
				TIMEVAL delta = tqhandle->next->expireTime - getCurrentTime();
				if (delta <= 0)
					delta = 1;
				setTimer(delta);
			}
		}
		queue_delete( tqhandle );
		
		// LeaveTimerMutex();
		
		MSG_DBG(handle,"DelAlarm", tqhandle->timer.id);
	}
	
	return TIMER_NONE;
}


void timerqueue_enable(void)
{
	memset(tq,0,sizeof(tq));
}

void queue_delete(register t_timerqueue *tq_old)
{
	if (tq_old->prev == NULL)
		return; // was already deleted

	if (tq_old->next != NULL)
	{
		tq_old->next->prev = tq_old->prev;
	}

	tq_old->prev->next = tq_old->next;

	tq_old->prev = /* tq_old->next = */ NULL;

}

void queue_insert(register t_timerqueue *tq_new)
{
	tq_new->prev = tq;
	tq_new->next = tq->next;
	
	while (tq_new->next != NULL 
			  && 
			  tq_new->expireTime - tq_new->next->expireTime > 0)
	{
		tq_new->prev = tq_new->next;
		tq_new->next = tq_new->next->next;
	}

	tq_new->prev->next = tq_new;

	if (tq_new->next != NULL)   /* in middle of queue */
	{
		tq_new->next->prev = tq_new;
	}
}


