/* simpleinit.c - poe@daimi.aau.dk */
/* Version 1.21 */

/* gerg@snapgear.com -- modified for direct console support DEC/1999 */

#define _GNU_SOURCE	/* For crypt() and termios defines */

#include <sys/types.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <pwd.h>
#include <sys/file.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/termios.h>
#include <sys/ioctl.h>
#include <sys/uio.h>
#include <linux/version.h>
#include <utmp.h>
#include <time.h>
#include <errno.h>
#include <termios.h>
#ifdef SHADOW_PWD
#include <shadow.h>
#endif
#include <linux/reboot.h>
//#if __GNU_LIBRARY__ > 5
#include <sys/reboot.h>
//#endif

//#include "hw_ram_rom.h"
#include "pathnames.h"

#define CMDSIZ 100	/* max size of a line in inittab */
#define NUMCMD 30	/* max number of lines in inittab */
#define NUMTOK 20	/* max number of tokens in inittab command */

#define TESTTIME  90	/* Threshold time for detecting "fast" spawning processes */
#define MAXSPAWN  5	/* Number of rapid respawns that counts as too fast */
#define MAXDELAY  595	/* Maximum delay between runs */
#define DELAYTIME 5	/* Time between successive runs of a process */

#define MAXTRIES 3	/* number of tries allowed when giving the password */

#define RUN_RC		/* Use a console if possible */
#define DEBUGGING 0

#undef HAVE_BOOT_SINGLE

#ifndef CTRL
#define CTRL(X) ((X)&037)
#endif

#ifdef INCLUDE_TIMEZONE
char tzone[CMDSIZ];
#endif
/* #define DEBUGGING */

/* Define this if you want init to ignore the termcap field in inittab for
   console ttys. */
/* #define SPECIAL_CONSOLE_TERM */

struct initline {
	pid_t	pid;
	time_t	lastrun;
	time_t	nextrun;
	char	*toks[NUMTOK];
	short	delay;
	char	tty[10];
	char	termcap[30];
	char	line[CMDSIZ];
	char	fullline[CMDSIZ];
	unsigned char xcnt;
};

struct initline inittab[NUMCMD];
int numcmd;
int stopped = 0;	/* are we stopped */
int reload = 0;	/* are we stopped */
int run_sigint_processing = 0;

extern void spawn(int);
extern void hup_handler(int);
extern void reload_inittab();
extern void read_inittab(void);
extern void read_initfile(const char *);
extern void tstp_handler(int);
extern void int_handler(int);
extern void sigint_processing();
extern void set_tz(void);
extern void make_ascii_tty(void);
extern void make_console(const char *);
extern int boot_single(int singlearg, int argc, char *argv[]);

/* Keep track of console device, if any... */
int have_console = 0;


static void err(const char *s)
{
	struct iovec output[2];
	output[0].iov_base =(char*)"init: ";
	output[0].iov_len = 6;
	output[1].iov_base = (void *)s;
	output[1].iov_len = strlen(s);
	if (have_console)
		writev(1, output, 2);
}
#ifdef HAVE_BOOT_SINGLE
static void enter_single(void)
{
	pid_t pid;
	char *av[2];

    err("Booting to single user mode\n");
    av[0] = _PATH_BSHELL;
    av[1] = NULL;
    if((pid = vfork()) == 0) {
    extern char **environ;
	/* the child */
	execve(_PATH_BSHELL, av, environ);
	err("exec of single user shell failed\n");
	_exit(0);
    } else if(pid > 0) {
    int i;
	while(wait(&i) != pid) /* nothing */;
    } else if(pid < 0) {
	err("fork of single user shell failed\n");
    }
    unlink(_PATH_SINGLE);
}
#endif


static int do_command(const char *path, const char *filename, int dowait)
{
	pid_t pid, wpid;
	int sts, st;

	if((pid = vfork()) == 0)
	{
		/* the child */
		char *argv[3];
#ifdef INCLUDE_TIMEZONE
		char tz[CMDSIZ];
#endif
		char *env[3];

		close(0);
		argv[0] = (char *)path;
		argv[1] = (char *)filename;
		argv[2] = NULL;

		env[0] = (char*)"PATH=/bin:/usr/bin:/etc:/sbin:/usr/sbin";
#ifdef INCLUDE_TIMEZONE
		strcpy(tz, "TZ=");
		strcat(tz, tzone);
		env[1] = tz;
		env[2] = NULL;
#else
		env[1] = NULL;
#endif

		execve(path, argv, env);

		err("exec rc failed\n");
		_exit(2);
	}
	else if(pid > 0)
	{
		if (!dowait)
			sts = 0;
		else {
			/* parent, wait till rc process dies before spawning */
			while ((wpid = wait(&sts)) != pid)
				if (wpid == -1 && errno == ECHILD) { /* see wait(2) manpage */
					sts = 0;
					break;
				}
		}
	} else if(pid < 0) {
		err("fork of rc shell failed\n");
		sts = -1;
	}
	st = WEXITSTATUS(sts);
	return st;
}

/*
 * run /etc/rc. The environment is passed to the script, so the RC environment
 * variable can be used to decide what to do. RC may be set from LILO.
 */
static int do_rc(void)
{
	int rc;
#ifdef HW_SDRAM_BASE
	char temp[256];
	/* start a special rc script for debug pupose only
	 * name of script is: rc-{last 4 bytes of the ethernet hardware address}
	*/
	sprintf(temp,"%s-%08lX",_PATH_RC,*((long*)(HW_SDRAM_BASE+0x402)));
	if(access(temp,0)==0)
		rc = do_command(_PATH_BSHELL, temp, 1);
	else
#endif
	rc = do_command(_PATH_BSHELL, _PATH_RC, 1);

	if (rc)
		return(rc);
#ifdef CONFIG_USER_INIT_RUN_FIREWALL
	rc = do_command(_PATH_FIREWALL, "-i", 1);
	if (rc)
		err(_PATH_FIREWALL " failed!");
#endif
#ifdef CONFIG_USER_FLATFSD_FLATFSD
	rc = do_command(_PATH_BSHELL, _PATH_CONFIGRC, 1);
	if (rc)
		err(_PATH_CONFIGRC " failed!");
#endif
#ifdef CONFIG_USER_INIT_RUN_FIREWALL
	rc = do_command(_PATH_FIREWALL, NULL, 0);
	if (rc)
		err(_PATH_FIREWALL " failed!");
#endif
#ifdef INCLUDE_TIMEZONE
	/* We read the timezone file here, because the flat file system
	 * has probably been created by now.
	 */
	set_tz();
#endif
	return(0);
}

void respawn_children(int used)
{
	int i, delta = -1;
	time_t now;
	alarm(0);
	if ((now = time(NULL)) == 0) now = 1;
	for(i = 0; i < numcmd; i++)
	{
		if(inittab[i].pid < 0)
		{	/* Start jobs */
			if(stopped)
				inittab[i].pid = -1;
			else
				spawn(i);
		}
		/* Check for naughty jobs */
		if (inittab[i].nextrun > now)
		{
		int d;
			d = inittab[i].nextrun - now;
			if (delta < 0 || d < delta)
				delta = d;
		}
	}
	if (delta > 0)
	{
		alarm(delta);
	}
}

int main(int argc, char *argv[])
{
	int 	i;
	int did_rc;
	struct sigaction sa;

	/*
	 * setup all the signal handlers here
	 */

	memset(&sa, 0, sizeof(sa));
	/* sa.sa_flags = SA_RESETHAND we want to keep the handlers armed */

	sa.sa_handler = int_handler;
	sigaction(SIGINT, &sa, NULL);

	sa.sa_handler = respawn_children;
	sigaction(SIGALRM, &sa, NULL);

	sa.sa_handler = hup_handler;
	sigaction(SIGHUP, &sa, NULL);


	/*
	 * start up in single user mode if /etc/singleboot exists or if
	 * argv[1] is "single".
	 */
#if defined(HAVE_SINGLE) && HAVE_SINGLE
	if(boot_single(0, argc, argv)) enter_single();
#endif
#ifdef RUN_RC
	/* Register console if defined by boot */
	struct stat st;

	if (isatty(1))
	{
		have_console = 1;
		make_ascii_tty();
	}
	else if (fstat(1, &st) == -1 && errno == EBADF)
	{
		close(0); close(1); close(2);
		open("/dev/null", O_RDWR);
		dup(0);
		dup(0);
	}

	/*If we get a SIGTSTP before multi-user mode, do nothing*/
	while(stopped)
		pause();
	did_rc=do_rc();

#ifdef HAVE_BOOT_SINGLE
	if(did_rc() !=	0 && boot_single(1, argc, argv) && !stopped )
		enter_single();
#endif
	while(stopped)	/*Also if /etc/rc fails & we get SIGTSTP*/
		pause();
#endif

	read_inittab();

#ifdef DEBUGGING
	for(i = 0; i < numcmd; i++) {
	char **p;
		p = inittab[i].toks;
		printf("toks= %s %s %s %s\n",p[0], p[1], p[2], p[3]);
		printf("tty= %s\n", inittab[i].tty);
		printf("termcap= %s\n", inittab[i].termcap);
	}
#endif

	i = 0;
	if (have_console)
		i = 3;
	for(; i < getdtablesize(); i++) close(i);

	for (;;)
	{
		pid_t	pid;
		int	vec;

		if (run_sigint_processing)
		{
			run_sigint_processing = 0;
			sigint_processing();
		}
		respawn_children(0);

		if (reload)
		{
			reload = 0;
			reload_inittab();
			continue; /* process all reloads before waiting */
		}

		pid = wait(&vec);
		alarm(0);

		/* clear utmp entry, and append to wtmp if possible */

		for(i = 0; i < numcmd; i++)
		{
			if(pid == inittab[i].pid)
			{
				inittab[i].pid = -1;
			}
		}
	}
}

#if  defined(HAVE_BOOT_SINGLE) && HAVE_BOOT_SINGLE
/*
 * return true if we should boot up in singleuser mode. If argv[i] is
 * "single" or the file /etc/singleboot exists, then singleuser mode should
 * be entered. If /etc/securesingle exists ask for root password first.
 */
int boot_single(int singlearg, int argc, char *argv[])
{
	char *pass, *rootpass = NULL;
	struct passwd *pwd;
	int i;

	for(i = 1; i < argc; i++) {
	    if(argv[i] && !strcmp(argv[i], "single")) singlearg = 1;
	}

	if(access(_PATH_SINGLE, 04) == 0 || singlearg) {
		if(access(_PATH_SECURE, 04) == 0) {
			if((pwd = getpwnam("root")) || (pwd = getpwuid(0)))
			  rootpass = pwd->pw_passwd;
			else
			  return 1; /* a bad /etc/passwd should not lock out */

			for(i = 0; i < MAXTRIES; i++) {
				pass = getpass("Password: ");
				if(pass == NULL) continue;

				if(!strcmp(crypt(pass, rootpass), rootpass)) {
					return 1;
				}

				puts("\nWrong password.\n");
			}
		} else return 1;
	}
	return 0;
}
#endif

void spawn(int i)
{
	pid_t pid;
	int j;
	time_t t;
	struct initline *it;
	char buf[150];

	it = inittab + i;

	t = time(NULL);
	if (it->nextrun > t)			/* Check for held process */
		return;
	if (it->lastrun + TESTTIME > t) {	/* Respawning quickly */
		if (it->xcnt < 0xff)
			it->xcnt++;
	} else {				/* Normal respawning */
		it->xcnt = 0;
		it->lastrun = t;
		it->delay = DELAYTIME;
	}
	if (it->xcnt >= MAXSPAWN) {		/* Too many too quickly */
		strcpy(buf, it->toks[0]);
		strcat(buf, " respawning too fast\n");
		err(buf);
		it->pid = -1;
		if (it->delay >= MAXDELAY)		it->delay = MAXDELAY;
		else if (it->delay < DELAYTIME)		it->delay = DELAYTIME;
		else if((it->delay *= 2) > MAXDELAY)	it->delay = MAXDELAY;
		it->nextrun = t + it->delay;
		/* Fiddle with the tracking vars to ensure that only
		 * one attempt is made to run this next time around.
		 */
		it->lastrun = it->nextrun;
		it->xcnt -= 2;
		return;
	}
	it->nextrun = t + DELAYTIME;

	if((pid = vfork()) < 0) {
		it->pid = -1;
		err("fork failed\n");
		return;
	}
	if(pid) {
		/* this is the parent */
		it->pid = pid;
		return;
	} else {
		/* this is the child */
		char term[40];
#ifdef INCLUDE_TIMEZONE
		char tz[CMDSIZ];
#endif
		char *env[4];

		setsid();

		// for(j = have_console ? 1 : 0; j < getdtablesize(); j++)
		for(j = 0; j < getdtablesize(); j++)
			close(j);
		make_console(it->tty);

		strcpy(term, "TERM=");
		strcat(term, it->termcap);
		env[0] = term;
		env[1] = (char*)"PATH=/var/bin:/bin:/usr/bin:/etc:/sbin:/usr/sbin";
#ifdef INCLUDE_TIMEZONE
		strcpy(tz, "TZ=");
		strcat(tz, tzone);
		env[2] = tz;
		env[3] = NULL;
#else
		env[2] = NULL;
#endif

		execve(it->toks[0], it->toks, env);
		strcpy(buf, it->toks[0]);
		strcat(buf, " exec failed\n");
		err(buf);
		_exit(1);
	}
}

static void init_itab(struct initline *p) {
	bzero(p, sizeof(struct initline));
	p->pid = -1;
}

void read_inittab(void)
{
	numcmd = 0;

	/* Fake an inittab entry if boot console defined */
#ifdef CONFIG_USER_INIT_CONSOLE_SH
	if (have_console)
	{
	struct initline *p;
		p = inittab + numcmd++;
		init_itab(p);
		strcpy(p->fullline, "console");
		strcpy(p->tty, "console");
		strcpy(p->termcap, "linux");
		p->toks[0] = "/bin/sh";
	}
#endif

	read_initfile(_PATH_INITTAB);

#ifdef CONFIG_USER_FLATFSD_FLATFSD
	read_initfile(_PATH_CONFIGTAB);
#endif

	if (numcmd == 0)
		_exit(1);
}

void read_initfile(const char *initfile)
{
	struct initline *p;
	FILE *f;
	char buf[CMDSIZ];
	int i,j,k;
	char *ptr, *getty;
#ifdef SPECIAL_CONSOLE_TERM
	char tty[50];
	struct stat stb;
#endif
	char *termenv;

	termenv = getenv("TERM");	/* set by kernel */
	/* termenv = "vt100"; */

	i = numcmd;

	if(!(f = fopen(initfile, "r"))) {
		err("cannot open inittab\n");
		return;
	}

	while(!feof(f) && i < NUMCMD - 2) {
		if(fgets(buf, CMDSIZ - 1, f) == 0) break;
		buf[CMDSIZ-1] = '\0';

		for(k = 0; k < CMDSIZ && buf[k]; k++) {
			if(buf[k] == '#') {
				buf[k] = '\0'; break;
			}
		}

		if(buf[0] == '\0' || buf[0] == '\n') continue;

		p = inittab + i;
		init_itab(p);
		strcpy(p->line, buf);
		strcpy(p->fullline, buf);
		ptr = strtok(p->line, ":");
		if (!ptr) {
			err("Missing TTY/ID field in inittab");
			continue;
		}
		strncpy(p->tty, ptr, 9);
		//p->tty[9] = '\0';
		ptr = strtok(NULL, ":");
		if (!ptr) {
			err("Missing TERMTYPE field in inittab");
			continue;
		}
		strncpy(p->termcap, ptr, 29);
		//p->termcap[29] = '\0';

		getty = strtok(NULL, ":");
		if (!getty) {
			err("Missing PROCESS field in inittab");
			continue;
		}
		strtok(getty, " \t\n");
		p->toks[0] = getty;
		j = 1;
		while((ptr = strtok(NULL, " \t\n")))
			p->toks[j++] = ptr;

#ifdef SPECIAL_CONSOLE_TERM
		/* special-case termcap for the console ttys */
		strcpy(tty, "/dev/");
		strcat(tty, p->tty);
		if(!termenv || stat(tty, &stb) < 0) {
			err("no TERM or cannot stat tty\n");
		} else {
			/* is it a console tty? */
			if(major(stb.st_rdev) == 4 && minor(stb.st_rdev) < 64) {
				strncpy(p->termcap, termenv, 30);
				p->termcap[29] = 0;
			}
		}
#endif

		i++;
	}
	fclose(f);

	numcmd = i;
}

void hup_handler(int d)
{
	reload = 1;
}

void reload_inittab()
{
	int i;
	int oldnum;
	char saveline[NUMCMD][CMDSIZ];
	pid_t savepid[NUMCMD];

	for (i=0; i<numcmd; i++) {
		savepid[i] = inittab[i].pid;
		strcpy(saveline[i], inittab[i].fullline);
	}
	oldnum = numcmd;
	read_inittab();

	/* See which ones still exist */
	for(i = 0; i < numcmd; i++) {
	int j;
		for(j = 0; j < oldnum; j++) {
			if(strcmp(saveline[j], inittab[i].fullline) == 0) {
				inittab[i].pid = savepid[j];
				savepid[j] = -1;
				break;
			}
		}
	}

	/* Kill off processes no longer needed */
	for(i = 0; i < oldnum; i++) {
		if (savepid[i] > 1)
			kill(savepid[i], SIGTERM);
	}
}


void int_handler(int d)
{
	run_sigint_processing = 1;
}

void sigint_processing()
{
	/*
	 * After Linux 0.96b PL1, we get a SIGINT when
	 * the user presses Ctrl-Alt-Del...
	 */

	int pid;

	sync();
	sync();

	if((pid = vfork()) == 0) {
		char *av[2];
		/* reboot properly... */
		av[0] = (char*)_PATH_REBOOT;
		av[1] = NULL;


		printf("init:shutdown\n");
		kill(-1, SIGHUP);
		sync();
		signal(SIGTERM,SIG_IGN);
		signal(SIGHUP,SIG_IGN);
		setpgrp();
		kill(-1, SIGTERM);
		sleep(1);
		kill(-1, SIGHUP);
		sleep(1);
		kill(-1, SIGKILL);

		do_command(_PATH_BSHELL, "/etc/rc.shutdown",1);
		sleep(1);

		reboot(LINUX_REBOOT_CMD_HALT);
		_exit(2);
	} else if(pid < 0) {
		/* fork failed, try the hard way... */
		reboot(LINUX_REBOOT_CMD_HALT);
	}
}

#ifdef INCLUDE_TIMEZONE
void set_tz(void)
{
	FILE *f;
	int len;

	if((f = fopen("/etc/config/TZ", "r")) == NULL &&
	   (f = fopen("/etc/TZ", "r")) == NULL)
		return;
	fgets(tzone, CMDSIZ-2, f);
	fclose(f);
	if((len=strlen(tzone)) < 2)
		return;
	tzone[len-1] = 0; /* get rid of the '\n' */
	setenv("TZ", tzone, 0);
}
#endif


void make_ascii_tty(void)
{
	struct termios tty;
	const char *pt;

	tcgetattr(0, &tty);
	tty.c_iflag &= ~(INLCR|IGNCR|IUCLC);
	tty.c_iflag |= ICRNL;
	tty.c_oflag &= ~(OCRNL|OLCUC|ONOCR|ONLRET|OFILL);
	tty.c_oflag |= OPOST|ONLCR;
	tty.c_cflag |= CLOCAL;
	tty.c_lflag  = ISIG|ICANON|ECHO|ECHOE|ECHOK|ECHOCTL|ECHOKE;
#ifdef IEXTEN
	tty.c_lflag  |= IEXTEN;
#endif


	tty.c_cc[VINTR]  = CTRL('C');
	tty.c_cc[VQUIT]  = CTRL('\\');
	tty.c_cc[VERASE] = CTRL('H'); /*127*/
	tty.c_cc[VKILL]  = CTRL('U'); /*Changed from non-standard ^X*/
	tty.c_cc[VEOF]   = CTRL('D');
	tty.c_cc[VTIME]  = 0;
	tty.c_cc[VMIN]   = 1;
	tty.c_cc[VSTART] = CTRL('Q');
	tty.c_cc[VSTOP]  = CTRL('S');
	tty.c_cc[VSUSP]  = CTRL('Z');
#ifdef VWERASE
	tty.c_cc[VWERASE]  = CTRL('W');
#endif
	/* Pick up simple environment setting of VERASE.
	 * Useful for setting on kernel command line.
	 * e.g. TTYERASE=^?
	 */
	pt = getenv("TTYERASE");
	if (pt && pt[0] == '^' && pt[1]) {
		tty.c_cc[VERASE] = (pt[1] == '?') ? 127 : CTRL(pt[1]);
	}

	tcsetattr(0, TCSANOW, &tty);
}

void make_console(const char *tty)
{
	int j;
	char devname[32];

	close(0); close(1); close(2);
	if (!tty || !*tty)
	{
		if (open("/dev/null", O_RDWR|O_NONBLOCK) >= 0)
			dup(0), dup(0);
		return;
	}

	{
		strcpy(devname, "/dev/");
		strcat(devname, tty);
	}

#if 1 // DAVIDM_FIXME
	if (open(devname, O_RDWR|O_NONBLOCK) == -1)
	{
#ifdef DEBUGGING
		printf("console '%s' open failed: %d\n", devname, errno);
#endif
		return;
	}
#endif

	fcntl(0, F_SETFL, 0);
	dup(0);
	dup(0);
	make_ascii_tty();
	j = ioctl(0, TIOCSCTTY, (char*)0);
}

