   <title>Python tutorial</title>
   <link rel="stylesheet" href="../markdown15.css">
   <script type="text/javascript" src="https://ajax.googleapis.com/ajax/libs/jquery/1.11.3/jquery.min.js"></script>
   <script type="text/javascript" src="../../../CreateTableOfContent.js"></script>
   <script type="text/javascript">
      $(document).ready(function () {
         CreateTableOfContent("table_of_content");
      });
   </script>
</head>
<table class="logo">
   <tr class="logo">
      <td style="text-align:right"><img style="height:54px" src="../../../images/sartorius-logo.png" alt="Sartorius Data Analytics" /></td>
   </tr>
   <tr style="height:20px"></tr>
</table>

Using Python in SIMCA
=====================

<div id="table_of_content">Table of content</div>

This tutorial gives an overview of how to use Python scripts in
SIMCA 15 and the functionality available.

Interacting with python in SIMCA
--------------------------------

The Python interpreter is embedded in SIMCA and can be used from
three different locations, the Add-Ins tab, Script favorites and
the Python console.

![SIMCA mainframe](../../../images/simca_frame.png)
*SIMCA with the Add-Ins tab, script favorites and Python console highlighted*

Script favorites are simple scripts that are executed when you
click on them. The [Add-Ins](../add-ins/add-ins.html) tab lets
you add command buttons to the ribbon, they are somewhat more
complicated to write but gives more flexibility. The Python
console executes commands as you type them much like the standard
Python console.<br />
For now we will use the Python console. If you can't see the
python console, press the "Python console" button on the Developer
tab.<br />
The rest of this guide assumes that you have some experience with
Python. If you don't, try one of the tutorials in the 'Python
tutorials' section of the 'Scripting appendix' in SIMCAs help file.

### A note on 'import'

In this tutorial and the example scripts, we will use fully
qualified names for methods, classes etc to make it clear where
methods, classes etc belongs. It is of course also possible to
import functionality using Pythons 'from' keyword to make the code
less verbose.

Example:<br />
SIMCA exports a package called 'umetrics' which contains a module
called 'impdata' which in turn contains a method called
'open_file_dialog'. To call this method we will always use:

```
import umetrics

umetrics.impdata.open_file_dialog()
```

But you could equally well use:

```
from umetrics.impdata import open_file_dialog

open_file_dialog()
```

Tutorial introduction
---------------------

### Hello World!
As is customary in programing tutorials we start with a "Hello
World!" example.<br />
Copy the lines below and paste them into the Python console.

```
from tkinter import Tk, messagebox
root=Tk()
root.withdraw()
messagebox.showinfo(title='Hello', message='Hello World!')
root.destroy()
```

You should see a message box that looks like this:

![Hello World!](../../../images/hello_world.png)

We will skip the details on how it works for now, suffice it to say that it uses a standard Python library called tkinter to popup a message box. All standard Python libraries are included in the SIMCA installation.

#### Turn the script into a favorite

We will now turn this simple script into a script favorite.<br />
Click the 'Create new script' on the 'Developer' tab. A 'Save As'
dialog will popup asking for a file name and a location for the
new script. Call the script 'hello_world' and save it in some
convenient location (like 'Documents'). An editor will open
containing the new script (the default editor is notepad). SIMCA
has added some boilerplate code that looks like this:

```
# The umetrics module contains all functionality that SIMCA exports to Python.
import umetrics
# The os module is a standard Python module that contains operating system functionality.
import os

def run():
    # Add code here

if __name__ == "__main__":
    run()
```

The script first imports two modules that contain functionality
that is useful in most scripts (in this example we won't use them
however). It then defines a method called 'run' in which we will
put our example code.<br />
The last two lines:

```
if __name__ == "__main__":
    run()
```

Is a very common Python idiom that controls when the 'run' method
is executed. We will not go into the details of why it is needed.
For an explanation see [this](http://www.effbot.org/pyfaq/tutor-what-is-if-name-main-for.htm).<br />
To start with we will create a new method that lets us popup a message box with any message.<br />
Before 'def run() :' enter:

```
from tkinter import Tk, messagebox

def show_message(caption, message):
    root= Tk()
    root.withdraw()
    messagebox.showinfo(title = caption, message = message)
    root.destroy()
```

We then need to call this method from the 'run' method. Modify the 'run' method so it looks like this:

```
def run() :
    show_message("Hello", "Hello World!")
```

The script should now look like [this](hello_world.html). Save the script. We now need to add the script to the Favorites tab.<br />
If you can't see the Favorites tab, check 'Favorites' on the View tab. Click 'Add script favorite' on the Developer tab.
Navigate to the directory where you saved the script and open it.<br />
The Favorites tab now has a favorite called 'hello_world'. If you click on it, the same message box will popup.

### Importing data

In this chapter we will write a script that creates a new project and imports data into it.
As in the 'Hello World!' example, create a new script but call it 'import', save it and add it to favorites.<br />
Modify the 'run' method so it looks like this:

```
def run() :
    (files, type) = umetrics.impdata.open_file_dialog(initialfile = 'Foods_predset.xls')
    if files == None :
        print('No file was selected')
        return
```

open_file_dialog is a method that pops up a file open dialog that has support for the file types that SIMCA can read.
The argument tells the method that 'Foods_predset.xls' should be the initial file name in the dialog. For the complete description see the [documentation](../technical/umetrics.impdata.html#-open_file_dialog).
Save the script and click the 'import' favorite. A dialog that looks like this will open:

![File Open Dialog](../../../images/file_open_dialog.png)

Press 'Cancel'. If you look in the python console you will see output that looks something like this:

```
>> # Executing 'C:\Users\Jerkero\Documents\import.py'
No file was selected
```

SIMCA will print the name of the favorite to the console when it runs, any output from the script will be also end up here, in this case the output from the print statement.
Error messages will also be printed to the console. If a script fails, output in the console can give valuable information on why it failed.<br />
Add this to the end of the 'run' function:

```
    if os.path.basename(files[0]) != 'Foods_predset.xls':
        print("This example only works with the file 'Foods_predset.xls'.")
        print("The file is installed with SIMCA and can normally be found in")
        print(r"C:\Program Files\Umetrics\SIMCA 15\Program\scriptexamples\data")
        return
```

Be careful with the indentation here. Python refuses to run scripts that aren't indented properly.<br />
'files' was returned by open_file_dialog and is a list of the file names selected. Since we only select one file the list will only contain one element.
os.path.basename is a standard python method that extracts the file name from a string containing the full path to a file.
The example we are about to write assumes that we use the 'Foods_predset.xls' file installed with SIMCA so we make sure that it is selected.<br />
This file can normally be found in 'C:\Program Files\Umetrics\SIMCA 15\Program\scriptexamples\data' or 'C:\Program Files (x86)\Umetrics\SIMCA 15\Program\scriptexamples\data'.<br />
It's time to read the file. Add this to the end of the 'run' function.

```
    data=umetrics.impdata.read_file(files, type)
```

The read_file method reads all file types that SIMCA knows how to read. For a complete list of supported file types, see the [documentation](../technical/umetrics.impdata.html#-read_file).
It returns an object of type [umetrics.impdata.ImportData](../technical/umetrics.impdata.html#ImportData) that holds the content of the file.
The arguments 'files' and 'type' was returned from open_file_dialog, 'files' is a list of the files to read (in this case only one) and
'type' is a string that helps read_file determine the type of the file. The 'type' argument can be omitted in which case read_file will try to identify the file type from the files content and extension.

#### Detour

If you need to import data from a source that isn't supported by read_file (like a database) you can create an ImportData object
and set its content from a script. Python has very good support for reading databases and various file formats.
This example show how to read an excel xml file and put the content into an ImportData object.
We will not go into details on how to read xml files.
The important thing here are the highlighted lines where we create an ImportData and inserts data from the xml file into it.

<pre style="margin-bottom:0;padding-bottom:0"><code># Reads an excel xml file and puts the data into a umetrics.impdata.ImportData and returns it.
def ReadXMLFile(file_name):
    # Name spaces used by the excel xml files we are reading
    xml_namespace = {'ns' : 'urn:schemas-microsoft-com:office:spreadsheet',
                     'ss' : 'urn:schemas-microsoft-com:office:spreadsheet'}
    tree=ET.parse(file_name)
    root = tree.getroot()
    table = root.find('./ns:Worksheet/ns:Table', xml_namespace)
    rows=int(table.attrib['{' + xml_namespace['ss'] + '}ExpandedRowCount'])
    columns=int(table.attrib['{' + xml_namespace['ss'] + '}ExpandedColumnCount'])</pre>
<pre style="margin-bottom:0;margin-top:0;padding: 0px 24px;"><code style="color:Crimson;">    import_data=umetrics.impdata.ImportData(rows, columns)</code></pre>
<pre style="margin-top:0;margin-bottom:0;padding: 0px 24px;"><code>    row_ix=0;
    for row in table.findall('ns:Row', xml_namespace):
        col_ix=0;
        for data in row.findall('ns:Cell/ns:Data', xml_namespace):</code></pre>
<pre style="margin-bottom:0;margin-top:0;padding: 0px 24px;"><code style="color:Crimson;">            import_data.set(row_ix, col_ix, data.text)</code></pre>
<pre style="margin-top:0"><code>        col_ix += 1
    row_ix += 1
    return import_data
</code></pre>

Now back to our example from this little detour.<br />

### Formating the data

Once we have read the file we also need to specify what is what (variable IDs, observation IDs etc.) in the data. The top left part of the file we will read looks like this:

![Spreadsheet](../../../images/foods.png)

The first column contains observation numbers which we will use as primary observation IDs,
the second column contains observation names which we will use as secondary observation IDs and
the first row contains the variable names (variable IDs).
To specify this information we use the 'importspec' member of the ImportData object. 'importspec' is of type
[umetrics.impdata.ImportSpecification](../technical/umetrics.impdata.html#ImportSpecification) which has methods for setting row and column types.
Add this to the end of the 'run' function:

```
    data.importspec.set_col_type(0, umetrics.impdata.ImportSpecification.columntype.primaryobsid)
    data.importspec.set_col_type(1, umetrics.impdata.ImportSpecification.columntype.secondaryobsid)
    data.importspec.set_row_type(0, umetrics.impdata.ImportSpecification.rowtype.primaryvarid)
```

It is now time to create a new project.<br />
We will put the new project in the directory where windows puts temporary files.
You can of course put new projects anywhere you like. <br />
Pythons standard library has a method, [filedialog.asksaveasfilename()](https://pythonspot.com/en/tk-file-dialogs/), that pops up a save file dialog that
you can use to let the user specify the name of the project and where it should be saved.<br />
To find out the name of the temporary files directory, we will use a standard python module called 'tempfile' which we must import.
Add 'import tempfile' to the top of the script so it looks like this.

```
# The umetrics module contains all functionality that SIMCA exports to Python.
import umetrics
# The os module is a standard Python module that contains operating system functionality.
import os
import tempfile
```

And add this to the end of the 'run' method.

```
    umetrics.simca.ProjectHandler.close_all_projects()
    project_file_name = os.path.join(tempfile.gettempdir(), "import_example.usp")
    if os.access(project_file_name, os.F_OK):
        os.unlink(project_file_name)
    project = umetrics.simca.ProjectHandler.create_project(project_file_name)
    project.create_dataset(data, "import example")
    project.save()
    umetrics.SimcaApp.set_active_project(project)
```

The complete example should now look like [this](import.html).<br />
The first line will close any projects that might be open in SIMCA.<br />
[ProjectHandler](../technical/umetrics.simca.html#ProjectHandler) is a class that is responsible for creating, opening, saving and closing projects.<br />
In the next three lines we create the file name we are going to use for the new project and delete it if it already exist.<br />
We then create the new project, create a dataset named 'import_example' with our data and save it.<br />
In the last line we tell SIMCA that our new project is the one that should be 'visible'.<br />
SIMCA can handle many projects at once but there is only one that the user can manipulate through the graphical user interface at any given time.
The class [SimcaApp](../technical/umetrics.html#SimcaApp) represents the visible part of SIMCA.<br />
That is it! If you click on the 'import' favorite and select  'Foods_predset.xls' the script will create the new project.


Python functionality overview
-----------------------------

The functionality SIMCA makes available to Python covers:

+ Reading supported file formats.
+ Creating new projects and datasets.
+ Create and fit new models.
+ Spectral filters.
+ Select predictionsets
+ Create plots and list
+ Obtaining results from models and predictions (scores, loadings etc)
+ Add command buttons to SIMCAs ribbon.
+ Add data vectors that will be available for plotting, coloring etc.

Note that this is by no means all functionality available to Python in SIMCA.
SIMCA comes with all standard Python libraries included and they cover things like database manipulation,
reading XML files, creating graphical user interfaces and much, much more.<br />
For a complete list see the [standard library documentation](https://docs.python.org/3.5/library/).
If you find that you need functionality that is not supported by the standard library it is very likely that
someone else has already written the code you need and made it available on the internet.<br />
A good place to look for libraries is the [Python wiki](https://wiki.python.org/moin/UsefulModules) but there are many others,
use your favorite search engine.

The Python functionality that SIMCA makes available is organized like this:

+ [umetrics](../technical/umetrics.html) package - The root package for all SIMCA functionality.
   + [SimcaApp](../technical/umetrics.html#SimcaApp) class - Represents the visible part of SIMCA. Has functionality for getting and setting the currently active project and creating plots and lists.
      + [PlotListBuilder](../technical/PlotListBuilder.html) class - Creates plots and lists
      + [GeneralOptions](../technical/GeneralOptions.html) class - Various options.
   + [SIMCAException](../technical/umetrics.html#SIMCAException) class - The exception class used by all functionality in the umetrics package.
   + [impdata](../technical/umetrics.impdata.html) package - Contains functionality for creating/reading and formatting data that can be imported as datasets in a project.
      + [ImportData](../technical/umetrics.impdata.html#ImportData) class - Used to hold data that can be imported as a dataset to a project.
      + [ImportSpecification](../technical/umetrics.impdata.html#ImportSpecification) class - When importing datasets, this class is used to specify how columns/rows in an ImportData should be interpreted (observation ids, qualitative variables etc.). Each ImportData object holds an instance of this class.
      + [open_file_dialog()](../technical/umetrics.impdata.html#-open_file_dialog) method - Opens a file open dialog that lets you select file(s) to read. Has support for all file formats that SIMCA can read.
      + [read_file()](../technical/umetrics.impdata.html#-read_file) method - Reads file(s) in one of the supported file formats and returns it as an ImportData object.
   + [simca](../technical/umetrics.simca.html) package - Contains functionality for manipulating projects, worksets, models, predictionsets, spectral filters and creating batch level datasets.
      + [BatchLevelCreator](../technical/umetrics.simca.html#BatchLevelCreator) class - Handles specification and creation of batch level datasets.
      + [DataCollection](../technical/umetrics.simca.html#DataCollection) class - Defines the complete data used by Datasets, prediction sets and work sets.
      + [Dataset](../technical/umetrics.simca.html#Dataset) class - Represents a SIMCA dataset. Inherits from DataCollection
      + [DatasetInfo](../technical/umetrics.simca.html#DatasetInfo) class - Contains information about a dataset.
      + [Filter](../technical/umetrics.simca.html#Filter) class - Spectral filter specification.
      + [ModelInfo](../technical/umetrics.simca.html#ModelInfo) class - Information about a model.
      + [ModelOptions](../technical/umetrics.simca.html#ModelOptions) class - Various options that effects coefficient types, DmodX etc.
      + [PhaseCropper](../technical/umetrics.simca.html#PhaseCropper) class - Defines trimming for phases in a workset.
      + [Predictionset](../technical/umetrics.simca.html#Predictionset) class - Defines observations that can be saved as a predictionset.
      + [Project](../technical/umetrics.simca.html#Project) class - The interface to SIMCA projects.
      + [ProjectData](../technical/umetrics.simca.html#ProjectData) class - Holds project data obtained from ProjectDataBuilder (below).
      + [ProjectDataBuilder](../technical/umetrics.simca.html#ProjectDataBuilder) class - Builds project data (scores, residuals, loadings etc) and returns them as a ProjectData.
      + [ProjectHandler](../technical/umetrics.simca.html#ProjectHandler) class - Contains functionality for creating, opening, saving and closing projects.
      + [ProjectOptions](../technical/umetrics.simca.html#ProjectOptions) class - Various batch options.
      + [UmMat](../technical/umetrics.simca.html#UmMat) class - A row major matrix of floating point numbers.
      + [UmVec](../technical/umetrics.simca.html#UmVec) class -  A vector of floating point numbers.
      + [ValueIDs](../technical/umetrics.simca.html#ValueIDs) class - Identifies values in a ProjectData.
      + [Workset](../technical/umetrics.simca.html#Workset) class - Defines the data, scaling, laging etc used to create a model.

The umetrics package is written in C++ so there is no python code available, instead the complete technical documentation can be accessed from the Developer tab in SIMCA.

![Help](../../../images/python_help.png)

There are also two classes written in Python that can be used to write SIMCA plugins.<br />
They are:

+ [RibbonAddInABC](../technical/ribbonabc.html): Used as base class to add command buttons to SIMCAs ribbon.
+ [ProjectVectorABC](../technical/projectvectorabc.html): Used as base class to add new vector types in SIMCAs plot/list dialogs.

Example scripts
---------------

### Introduction

In this section we will have a look at the contents of the example scripts that come pre-installed with SIMCA.
These scripts are fully functional and can be run repeated times with the same result.
All data required for the scripts are also pre-installed.

![](../../../images/example_scripts.png)

The example scripts are very useful for understanding how to interact with the SIMCA functionality through the Python interface.
They are also good sources of useful code bits that you can copy to your own scripts.
Even better, make a copy of the example script that looks most like the solution you want and then start modifying the copy.

When using the scripts as sources for code bits it is important to remember that some steps of certain procedures are
omitted in some of the scripts.
The scripts still work because there are a number of default settings and selections that will be used if nothing is specified.
The defaults are the same as inside SIMCA so for instance, when you create a new [ImportData](../technical/umetrics.impdata.html#ImportData) object,
all columns with numbers are assumed to be quantitative variables.
This means that you only need to specify the columns that should be used in some other way, e.g. as an ID column.
Another example is when you create a model for a workset you only need to indicate the type of model if it is not the default type,
e.g. pcaX is default when you only have X-variables and pls when you have both X and Y.

Below follows a short description of what each of the example script does and then we will have a more
detailed look at how certain procedures are used in the different scripts.

### Overview of the example scripts

#### Create project ([createproject.py](createproject.html))

This script shows how to perform the basic procedures in SIMCA.
It shows how to import data from an EXCEL-file, create a project, dataset, workset, and model.
It then fits a 3-component PLS model and creates a score scatter plot.

#### Create spectral project ([createspectralproject.py](createspectralproject.html))

This script imports all spectral SPC-files from a folder (specified in the Python code),
creates a new project, filters the raw spectra using SNV and creates a new dataset with filtered data.
The script then creates a workset from the filtered data, fits a PCA model using Autofit, and displays a score scatter plot.

#### Import predictionset ([importpredset.py](importpredset.html))

This script opens an existing project (specified in the Python code) and then imports additional data from an EXCEL-file.
It then creates a predictionset from the new data and projects the observations using the active (PCA) model.
It then creates two plots, a column plot with DModXPS and a scatter plot with the tPS[1] score values for the predictionset.

#### Complement batch project ([complementbatch.py](complementbatch.html))

This script shows how you can add new batch data to an existing batch project.
The script opens an existing batch project (specified in the Python code),
imports three new batches from the same DIF file, creates a copy of the existing BEM (phase) models and
adds the new batches to the model copies. It then fits all five BEM (phase) models.

#### Enable ribbon plugin examples

This example script will not be described here.<br />
There is separate tutorial on how to create [plug-ins](../add-ins/add-ins.html).<br />
It is different than all the other scripts in that it
does not perform any modelling or predictions. It only shows how to create buttons in the "Add-In" tab.
This script can also only be run once. The second time you run it, nothing happens.

#### Import functionality in the example scripts

In the example scripts we show how to import data with the "read-file" function from EXCEL-files, DIF-files and CSV-files.
Different file formats have different (optional) import options and in the example scripts we show how to use
XML-strings to set some of the specific import options.
A complete list of available import options can be found [here](../technical/umetrics.impdata.html#-read_file).
Below follows some illustrations of the different import examples and links to where to find them.

#### Import data from EXCEL-files

In the [Create project](tutorial.html#Create_project_(createproject.py)) script it is shown how to use import options to specify which EXCEL worksheet to import the data from.
This script uses the import options to be able to use data from two different worksheets.
The import of an EXCEL-file with options is also used in the [Import predictionset](tutorial.html#Import_predictionset_(importpredset.py)) script.
A complete list of options for EXCEL-files can be found [here](../technical/umetrics.impdata.html#-read_file).

```
# Create a setting to read first two sheets from the Excel file
# UseIndexes tells that the index of the sheets should be used instead of the sheet name
optionsSheet1 = '<Settings Sheets="0" UseIndexes="1"/>'
optionsSheet2 = '<Settings Sheets="1" UseIndexes="1"/>'
# Read the data sheets
sheet1 = umetrics.impdata.read_file(datapath, openoptions = optionsSheet1)
sheet2 = umetrics.impdata.read_file(datapath, openoptions = optionsSheet2)
```

#### Import data from Galactic SPC-files

In the example script [Create spectral project](tutorial.html#Create_spectral_project_(createspectralproject.py)) we show how to import multiple spectral files at the same time.
This script not only shows how to import Galactic SPC-files but also how to import multiple files at the same time.
To do this we create a list of all the files we want to import by using the command "glob.glob" which will return
a list of the files fulfilling the criteria (*.spc) in the specified data folder.

```
# The paths for data files and where the project should be saved
scriptpath = os.path.dirname(os.path.realpath(__file__))
datapath = os.path.join(scriptpath, 'data/*.spc')

# Read all spectral files (*.spc) from a directory
files = glob.glob(datapath)
```

There are many import options that can be set for SPC-files (see [documentation](../technical/umetrics.impdata.html#-read_file))
and putting them all in one XML string can become hard to overview,
therefore we use functions from the XML ElementTree module.
In the body of the script we then create an XML tree with the options and in the "read_file" command
convert the tree structure to a string. An alternative to this is to omit all options which will result in
SIMCA displaying a dialog for the options.

```
# Create an XML tree with the spc open options.
# If we don't supply these options, SIMCA will popup a dialog and ask for them.
spcoptions=ET.Element('Settings')
spcoptions.set('IncludeOrdinalValues', '1')
spcoptions.set('IncludeAllValues', '1')
spcoptions.set('IncludeXStartPt', '')
spcoptions.set('IncludeXEndPt', '')
spcoptions.set('NumberOfIncludedIntervals', '0')
spcoptions.set('UseAverage', '0')

data = umetrics.impdata.read_file(files, openoptions=ET.tostring(spcoptions))
```

#### Import data from DIF-files

The import of DIF files is very straightforward and has a lot in common with the import of CSV or TXT files.
In the [Complement batch project](tutorial.html#Complement_batch_project_(complementbatch.py)) script we only need to provide the file name and the encoding definition.
The different encoding options are listed [here](../technical/umetrics.impdata.html#-read_file).

```
importOptions = '<Settings Encoding="windows1252"/>'
datafile = umetrics.impdata.read_file(datapath, openoptions = importOptions)
```

#### Interacting with the [ImportData](../technical/umetrics.impdata.html#ImportData) object

An important step between the original files and the dataset inside SIMCA is the ImportData object
that is created by the "read_file" function in all of the example scripts.
In an earlier tutorial [section](#Importing_data),
alternative ways to create the ImportData object is described (e.g. using a dialog or creating an empty
ImportData object of a given size). Regardless of how you created the object, you can (and need to) interact with it.

##### Setting primary ID:s

The minimum interaction with the ImportData object you need to perform is to specify where you have your
Primary ID:s (observation and variable). This can look as it does in the "Create predictionset" script seen below.

```
# Specify primary and secondary ID
# Set first row to primary variable ID
datafile.importspec.set_row_type(0, umetrics.impdata.ImportSpecification.rowtype.primaryvarid)
# Set first colum to primary observation ID
datafile.importspec.set_col_type(0, umetrics.impdata.ImportSpecification.columntype.primaryobsid)
```

If you are working with a batch project you also need to specify a BatchID column and also a PhaseID column if you have phase models.
The setting of Batch and Phase ID is exemplified in the [Complement batch project](tutorial.html#Complement_batch_project_(complementbatch.py)) script together with how to exclude columns
in the ImportData object.

Other types of data properties and uses you can define for rows and columns in an ImportData object can be found [here](../technical/umetrics.impdata.html#ImportSpecification).

##### The "get" and "set" commands to modify values

Before creating a dataset from the ImportData object you can also perform your own pre-treatments,
normalizations, transformation, or other calculations with the data in the ImportData object.
You can retrieve the data from the ImportData object by using a "get" command and when you have your new modified values
you can push them in to the ImportData object using a "set" command. To use any of these commands you need to provide the correct location to the "set" or "get" command
(i.e. cell position specified by row and column numbers).
In [Create project](tutorial.html#Create_project_(createproject.py)) we want to introduce the string "Wave Number" as the name of the secondary ID. To accomplish this,
we use the "set" command to insert the string in the correct cell of the ImportData object called "data".

```
# Set the first column in the second row to 'Wave Number'. Then the secondary ID will get the correct name
data.set(1,0, 'Wave Number')
```

#### Creating project, dataset, predictionset, workset and model

##### Project and Dataset

Before we can convert an ImportData object into a dataset that we can use inside SIMCA,
we need to create the project object that will serve as the container for them.
Once we have a Project, we can create the dataset and we give them names.
In the [Create project](tutorial.html#Create_project_(createproject.py)) example script we first create the Project object (here called "project")
and then we create two separate datasets inside that container and call them "DS1" and "DS2"

```
# Create a new project
project = umetrics.simca.ProjectHandler.create_project(projectpath)

# Create the dataset
ds_num1 = project.create_dataset(sheet1, "DS1")
ds_num2 = project.create_dataset(sheet2, "DS2")
```

##### Predictionset

When you want to make predictions, you have to create and save a Predictionset which is very similar to a Dataset.
A predictionset can be created directly from a Dataset as shown in the [Import predictionset](tutorial.html#Import_predictionset_(importpredset.py)) script where
"ds_num1" is the Dataset number for the source Dataset.

There are other ways of creating a predictionset, including selecting individual observations by using the "add_source"
command. A complete list of how to create a predictionset can be found [here](../technical/umetrics.simca.html#Predictionset).

```
# Create a prediction set using the dataset for the active model
active_model = project.get_active_model()

# Create a new predictionset from the dataset
predset = project.create_predictionset(active_model)
predset.as_dataset(ds_num1)
predset.save("MyPredset")

# Set predictionset as current predictionset to use
project.set_predictionset("MyPredset")
```

It is important to remember to "save" and then "set" the predictionset as the active one in order to be able to use it.

##### Applying spectral filters to data

To use the filter functionality in SIMCA you will have to start by creating a filter object.
In the [Create spectral project](tutorial.html#Create_spectral_project_(createspectralproject.py)) script we show how to create, define, and apply an SNV filter.

Note in the first row of code below that the filter object is created using a dataset number (or name) as an input argument.
You then proceed by defining the filter type as one of the [available filter types](../technical/umetrics.simca.html#filtertype).
When you then apply this filter you provide the name you want to use on the filtered dataset.

```
# Filter the dataset using SNV
filter=project.create_filter(ds_num)
filter.add_filter_type(umetrics.simca.filtertype.snv)
filter.apply_filter('snv_filtered')
```

There are a number of settings associated with different types of filters and those can be set
as attributes for the filter object.
Different filter types require different attributes,
but all available filter settings can be found [here](../technical/umetrics.simca.html#Filter) and if not specified, the normal defaults for SIMCA will be used.

##### Workset and model

As inside SIMCA, you need to define a workset before you can fit a model.
There are however some important steps you need to include in the scripts to be able to create the models you intend.
In all the example scripts (except [Import predictionset](tutorial.html#Import_predictionset_(importpredset.py))) we go through the procedures of defining a workset
and fitting a model. Some steps are used in all scripts but in addition,
each script contain some specific optional procedure that is good to know.

*Creating the workset*

First step is to create a workset from one or more data sources (datasets).
The created workset will be the default workset associated with the dataset.
In [Create spectral project](tutorial.html#Create_spectral_project_(createspectralproject.py)) we show the simplest situation where you only provide one dataset name.

```
# Create a workset for the model
workset = project.create_workset('snv_filtered')
```

In [Create project](tutorial.html#Create_project_(createproject.py)) we show how to create a workset from more than one dataset and here we have
chosen to use dataset numbers instead of names.

```
# Create a workset for the model using the two datasets
workset = project.create_workset([ds_num1, ds_num2])
```

In the [Complement batch project](tutorial.html#Complement_batch_project_(complementbatch.py)) we create the workset based on a previous workset.
Normally you provide the workset number of the workset you want to copy but in this case we use the minus
("-") prefix to indicate that it is a model group.
In this specific case it is the batch evolution model group with all 5 phase models.

```
# Create a new workset from the first model group that we want to complement with new batches
workset = project.new_as_workset(-1)
```

*Interacting with the workset*

Once a workset has been created, it can be interacted with and in the example scripts we show some of the
available modifications you can perform. There are many possible actions you can perform on the workset and the
best way of describing it is to say that everything you can do in the workset dialog inside SIMCA
you can also do using scripting. See the complete list [here](../technical/umetrics.simca.html#Workset).

In [Create project](tutorial.html#Create_project_(createproject.py)) we check the number of variables in the workset and sets the last variable to be a y-variable.

```
# Set the last variable to Y
numVars = workset.get_data_collection().size_variables()
workset.set_y(numVars - 1)
```

In [Create spectral project](tutorial.html#Create_spectral_project_(createspectralproject.py)) we change the scaling type for all variables (the empty list "[ ]" means all variables).
The available scaling types can be found [here](../technical/umetrics.simca.html#scaletype).
We also exclude some variables from the workset.

```
# Change scaling to center for all variables
workset.set_variable_scale_type([], umetrics.simca.scaletype.ctr)

# Exclude the first three variables
workset.exclude_variables([0, 1, 2])
```

In [Complement batch project](tutorial.html#Complement_batch_project_(complementbatch.py)) we add an additional dataset to an existing workset.

```
# Add the new dataset with the new batches
workset.add_dataset(ds_num)
```

*Setting the type of model*

In order to fit a model we first need to define the type of model we want to use on the workset.
This is only exemplified in the [Create project](tutorial.html#Create_project_(createproject.py)) script since it is not necessary to specify explicitly.
If you do not specify any model type, the default model type will be used. (e.g. when only x-variables are defined,
pcaX is the default. With x- and y-variables defined, pls is the default.)<br>
See [umetrics.simca.modeltype](../technical/umetrics.simca.html#modeltype) for details.

```
# Set the model typ to PLS
workset.set_type(umetrics.simca.modeltype.pls)
```

*Creating and fitting the model*

To be able to fit a model we first need to create a model from the workset.
Only defining the type of model as shown above is not enough.
The creation of the model object is done in the same way regardless of project.
The fitting of the models however, depends on the workset, model, and modelling objective.
Some alternatives are exemplified in the example scripts but a complete list of arguments
can be found in the [documentation](../technical/umetrics.simca.html#Project-fit_model).
In the [Create project](tutorial.html#Create_project_(createproject.py)) script we create the model and fit a three component model.

```
# Create the model and fit with three components
model = workset.create_model()
project.fit_model(model[0], numcomp=3)
```

The model argument "[0]" indicates the first model in the list of models.
The list contains the model numbers of the created models so calling on "model[0]" in this case will
return the model number of the created model. In this example, and in many other applications,
the list only contains one model number but as can be seen in the later description of [Complement batch project](tutorial.html#Complement_batch_project_(complementbatch.py))
the model list can actually contain several models.

In [Create spectral project](tutorial.html#Create_spectral_project_(createspectralproject.py)) we do not provide any argument for the number of components. 
SIMCA will then use autofit. The same is achieved if you use "numcomp = -1".

```
# Create the model and autofit
model = workset.create_model()
project.fit_model(model[0])
```

In the [Complement batch project](tutorial.html#Complement_batch_project_(complementbatch.py)) example script the creation of the model object results in a model group. 
The model group list in this example contains all five phase evolution model numbers. 
It is therefore necessary to loop over all model numbers in the list to fit all models.

```
# Create and fit the models
models = workset.create_model()
for modelNumber in models :
project.fit_model(modelNumber)
```

#### Generating plots

When you have your models you may want to generate some plots. 
From the scripting environment you can access all default plots in SIMCA. 
This means that when you call for a score scatter plot in your script you will get the t[1]/t[2] plot 
from a PCA model tp[1]/to[1] from an OPLS model. 
With the current implementation, these displayed vectors cannot be changed using a script.

To create plots for the active project you first need to create an instance of the "PlotListBuilder" 
and then you can create all the plots listed in the [PlotListBuilder](../technical/PlotListBuilder.html) 
class documentation.

In the [Create project](tutorial.html#Create_project_(createproject.py)) and [Create spectral project](tutorial.html#Create_spectral_project_(createspectralproject.py)) we create a "builder" object and then create the score scatter plot.

```
# Create Summary of fit plot and score scatter plot
builder = umetrics.SimcaApp.plot_list_builder()
builder.score_scatter_plot()
```

When running any of these scripts you most likely also will see the "Summary of Fit" plot. 
This plot is not generated by the "builder" object above. It is a plot that is automatically generated 
when fitting a model. You can turn this feature off in the SIMCA options (File | Options | SIMCA options). 

In the [Import predictionset](tutorial.html#Import_predictionset_(importpredset.py)) script we also show how to create a DModXPS plot. 
To correctly display the "Distance to model" you need to specify the type of "Distance to model" you want. 
See [documentation](../technical/PlotListBuilder.html#PlotListBuilder-yrelated_profiles_overview) for a full list.

```
builder.dmod_column_plot(umetrics.SimcaApp.PlotListBuilder.DModType.DModXPS)
```

#### Generating and Accessing Vector Data
Similar to the [PlotListBuilder](../technical/PlotListBuilder.html), the [ProjectDataBuilder](../technical/umetrics.simca.html#ProjectDataBuilder)
can be used to create and access vector data in python. The ProjectDataBuilder can be accessed through the active Project object:
```
# Assuming the project from import prediction set example, Importpredset.usp is active
project = umetrics.SimcaApp.get_active_project()
projectDataBuilder = project.data_builder()
```
The create function provided by the ProjectDataBuilder can be used to access a wide range of [available vectors](../technical/umetrics.simca.html#ProjectDataBuilder-create).
For example the *'t'* vector is very commonly used. After running the [importpredset.py](importpredset.html) example, having the imortpredset project active we can create the *t* vector for it like this:
```
pd = projectDataBuilder.create("t", model = 1)
```
The workset spreadsheet for Importpredset has two additional observation name columns to the primary id.<br>
![Importpredset.usp](../../../images/worksheet.png)

*In the spreadsheet the column indices start at 1. This is reflected in the project data value ids.*

| ProjectData Methods | Return Values |
|---------|----------------|
| matrix() | Multi-dimensional array of vector data |
|series_names() | List of names of the data series in the matrix data |
|size_series_aliases() | The number of series names? |
|get_value_ids() | A [ValueIds](../technical/umetrics.simca.html#ValueId) object with methods size_aliases() and get_names() |
|size_aliases() | The number of name lists |
|get_names(*index*) | List of names for the given index, or the primary id list if no argument is passed |

```
vids = pd.get_value_ids()        
pd.size_series_aliases() 1
pd.series_names()        ['M1.t[1]']
vids.size_aliases()      3  #indexing for get_names() starting at 1
vids.get_names(1)        [      '1',     '2',      '3',       '4',       '5',       '6',      '7',       '8',       '9']
vids.get_names(2)        [      '1',     '2',      '3',       '4',       '5',       '6',      '7',       '8',       '9']
vids.get_names(3)        ['Germany', 'Italy', 'France', 'Holland', 'Belgium','Luxembou','England','Portugal', 'Austria']
pd.matrix()             [['-1.9347','3.6232', '0.6098', '-2.8208',  '0.7830', '-2.4897','-4.2785',  '3.8999',  '2.6078']]
```

This corresponds to selecting one series List in Plot/List

![General List UI](../../../images/general_list.png)

The generated list in the GUI for comparison

![General List M1](../../../images/general_list_t.png)


*CSV*

In reality, printing values to the console output is not very efficient or practical. CSV is a simple file format that is interoperable with Microsoft Excel and many other applications and tools.
To write data to a CSV file, the [csv](https://docs.python.org/3.5/library/csv.html) package in python can be used.
```
import csv
```
In order to avoid empty lines in the output file, the *newline=''* argument to the open function call is needed.
```
file_name = r'C:\Windows\Temp\ex.csv'
with open(file_name, "w", newline='') as csvfile :
    writer = csv.writer(csvfile)
    for row in pd.matrix() :
        writer.writerow(row)
```
This will write the names and matrix data horizontally. Each name list and matrix entry as a row.
Often this is not the most convenient way to export the data. A more practical way would be to have
the name lists and matrix data in columns.
The following example rotates the orientation of the data 90 degrees:
```
def make_rows(pd):
     vids = pd.get_value_ids()
     ran  = range(1, vids.size_aliases() + 1)
     names = [vids.get_names(r) for r in ran]
     vertical = zip(*names, *pd.matrix())
     return vertical
```

The output from this function
```
for p in make_rows(pd):
    print(p)

('1', '1', 'Germany', -1.9346758127212524)
('2', '2', 'Italy', 3.6232006549835205)
('3', '3', 'France', 0.6098378896713257)
('4', '4', 'Holland', -2.8208000659942627)
('5', '5', 'Belgium', 0.7830119132995605)
('6', '6', 'Luxembou', -2.4896926879882812)
('7', '7', 'England', -4.278542995452881)
('8', '8', 'Portugal', 3.8998794555664062)
('9', '9', 'Austria', 2.6077816486358643)
```

To write it to a csv file:

```
file_name = r'C:\Windows\Temp\vert.csv'
with open(file_name, "w", newline='') as csvfile :
    writer = csv.writer(csvfile)
    for row in make_rows(pd) :
        writer.writerow(row)
```

Will result in the following layout in excel

![C:\Windows\Temp\vert.csv](../../../images/csv_exported_t.png)

#### Model Information View Object
Information about the models can be accessed through the Project object with get_model_infos() method. 
It returns a list of immutable ModelInfo objects. The list starts with the first model, usually 1,
but it can be 2, if the first has been removed, or -1 if it is a batch project.
For example, with the BatchProject.usp from the [Complement batch project](tutorial.html#Complement_batch_project_(complementbatch.py))
example script open, the first model has the number -1. It's the mother model for the batch models.
```
>>> project = umetrics.SimcaApp.get_active_project()
>>> infos = project.get_model_infos()
>>> print(infos)
[(-1,BM1,...), (1,M1:1,...), (2,M2:2,...), (3,M3:3,...), (4,M4:4,...), (5,M5:5,...)]
```
This short string representation shows that there are six models. Looking at the details of the first one:
```
>>> print(infos[0])
number        = -1
name          = 'BM1'
description   = 'Untitled
type          = plsClass
created       = Thu Jun 12 10:31:26 2014 UTC
observations  = 212
Xvariables    = 16
Yvariables    = 1
lagged        = 0
expanded      = 0
isfitted      = False
components    = 0
orthogonalinX = 0
orthogonalinY = 0
mother        = 0
children      = [1, 2, 3, 4, 5]
```
The properties number, mother and children are related. In batch projects there is a mother model which has the number -1 and one or more entries in its children.
The child models have children set to an empty list and mother set to -1. For regular projects there is no mother. All models have mother set to 0 and children to empty list.

Most properties are simple types such as numbers, strings and dates. The type field is an internal enum [umetrics.simca.modeltype](../technical/umetrics.simca.html#modeltype) that is described in the documentation.

The properties of the ModelInfo object can be as a normal member variable but can not be modified. For example to check for existence of certain data in a model:
```
if info.orthogonalinY > 0:
    pd = projectDataBuilder.create("co", model=info.number)
```
The 'co' vector is not applicable if there are no orthogonals in Y.

<table class="copyright">
   <tr style="height:20px"></tr>
   <tr class="copyright"><td>Copyright&copy; Sartorius Stedim Data Analytics AB</td></tr>
   <tr class="copyright"><td>SIMCA, part of the Umetrics&#174; Suite of Data Analytics Solutions</td></tr>
   <tr class="copyright"><td>SIMCA&#174; Multivariate Data Analytics Solution</td></tr>
</table>